/*

	Speed Plug-in

	

	Features:

		*Adds several properties to the 'file' object indicated upload speed, time left, upload time, etc.

			- currentSpeed -- String indicating the upload speed, bytes per second

			- averageSpeed -- Overall average upload speed, bytes per second

			- movingAverageSpeed -- Speed over averaged over the last several measurements, bytes per second

			- timeRemaining -- Estimated remaining upload time in seconds

			- timeElapsed -- Number of seconds passed for this upload

			- percentUploaded -- Percentage of the file uploaded (0 to 100)

			- sizeUploaded -- Formatted size uploaded so far, bytes

		

		*Adds setting 'moving_average_history_size' for defining the window size used to calculate the moving average speed.

		

		*Adds several Formatting functions for formatting that values provided on the file object.

			- SWFUpload.speed.formatBPS(bps) -- outputs string formatted in the best units (Gbps, Mbps, Kbps, bps)

			- SWFUpload.speed.formatTime(seconds) -- outputs string formatted in the best units (x Hr y M z S)

			- SWFUpload.speed.formatSize(bytes) -- outputs string formatted in the best units (w GB x MB y KB z B )

			- SWFUpload.speed.formatPercent(percent) -- outputs string formatted with a percent sign (x.xx %)

			- SWFUpload.speed.formatUnits(baseNumber, divisionArray, unitLabelArray, fractionalBoolean)

				- Formats a number using the division array to determine how to apply the labels in the Label Array

				- factionalBoolean indicates whether the number should be returned as a single fractional number with a unit (speed)

				    or as several numbers labeled with units (time)

	*/



var SWFUpload;

if (typeof(SWFUpload) === "function") {

	SWFUpload.speed = {};

	

	SWFUpload.prototype.initSettings = (function (oldInitSettings) {

		return function () {

			if (typeof(oldInitSettings) === "function") {

				oldInitSettings.call(this);

			}

			

			this.ensureDefault = function (settingName, defaultValue) {

				this.settings[settingName] = (this.settings[settingName] == undefined) ? defaultValue : this.settings[settingName];

			};



			// List used to keep the speed stats for the files we are tracking

			this.fileSpeedStats = {};

			this.speedSettings = {};



			this.ensureDefault("moving_average_history_size", "10");

			

			this.speedSettings.user_file_queued_handler = this.settings.file_queued_handler;

			this.speedSettings.user_file_queue_error_handler = this.settings.file_queue_error_handler;

			this.speedSettings.user_upload_start_handler = this.settings.upload_start_handler;

			this.speedSettings.user_upload_error_handler = this.settings.upload_error_handler;

			this.speedSettings.user_upload_progress_handler = this.settings.upload_progress_handler;

			this.speedSettings.user_upload_success_handler = this.settings.upload_success_handler;

			this.speedSettings.user_upload_complete_handler = this.settings.upload_complete_handler;

			

			this.settings.file_queued_handler = SWFUpload.speed.fileQueuedHandler;

			this.settings.file_queue_error_handler = SWFUpload.speed.fileQueueErrorHandler;

			this.settings.upload_start_handler = SWFUpload.speed.uploadStartHandler;

			this.settings.upload_error_handler = SWFUpload.speed.uploadErrorHandler;

			this.settings.upload_progress_handler = SWFUpload.speed.uploadProgressHandler;

			this.settings.upload_success_handler = SWFUpload.speed.uploadSuccessHandler;

			this.settings.upload_complete_handler = SWFUpload.speed.uploadCompleteHandler;

			

			delete this.ensureDefault;

		};

	})(SWFUpload.prototype.initSettings);



	

	SWFUpload.speed.fileQueuedHandler = function (file) {

		if (typeof this.speedSettings.user_file_queued_handler === "function") {

			file = SWFUpload.speed.extendFile(file);

			

			return this.speedSettings.user_file_queued_handler.call(this, file);

		}

	};

	

	SWFUpload.speed.fileQueueErrorHandler = function (file, errorCode, message) {

		if (typeof this.speedSettings.user_file_queue_error_handler === "function") {

			file = SWFUpload.speed.extendFile(file);

			

			return this.speedSettings.user_file_queue_error_handler.call(this, file, errorCode, message);

		}

	};



	SWFUpload.speed.uploadStartHandler = function (file) {

		if (typeof this.speedSettings.user_upload_start_handler === "function") {

			file = SWFUpload.speed.extendFile(file, this.fileSpeedStats);

			return this.speedSettings.user_upload_start_handler.call(this, file);

		}

	};

	

	SWFUpload.speed.uploadErrorHandler = function (file, errorCode, message) {

		file = SWFUpload.speed.extendFile(file, this.fileSpeedStats);

		SWFUpload.speed.removeTracking(file, this.fileSpeedStats);



		if (typeof this.speedSettings.user_upload_error_handler === "function") {

			return this.speedSettings.user_upload_error_handler.call(this, file, errorCode, message);

		}

	};

	SWFUpload.speed.uploadProgressHandler = function (file, bytesComplete, bytesTotal) {

		this.updateTracking(file, bytesComplete);

		file = SWFUpload.speed.extendFile(file, this.fileSpeedStats);



		if (typeof this.speedSettings.user_upload_progress_handler === "function") {

			return this.speedSettings.user_upload_progress_handler.call(this, file, bytesComplete, bytesTotal);

		}

	};

	

	SWFUpload.speed.uploadSuccessHandler = function (file, serverData) {

		if (typeof this.speedSettings.user_upload_success_handler === "function") {

			file = SWFUpload.speed.extendFile(file, this.fileSpeedStats);

			return this.speedSettings.user_upload_success_handler.call(this, file, serverData);

		}

	};

	SWFUpload.speed.uploadCompleteHandler = function (file) {

		file = SWFUpload.speed.extendFile(file, this.fileSpeedStats);

		SWFUpload.speed.removeTracking(file, this.fileSpeedStats);



		if (typeof this.speedSettings.user_upload_complete_handler === "function") {

			return this.speedSettings.user_upload_complete_handler.call(this, file);

		}

	};

	

	// Private: extends the file object with the speed plugin values

	SWFUpload.speed.extendFile = function (file, trackingList) {

		var tracking;

		

		if (trackingList) {

			tracking = trackingList[file.id];

		}

		

		if (tracking) {

			file.currentSpeed = tracking.currentSpeed;

			file.averageSpeed = tracking.averageSpeed;

			file.movingAverageSpeed = tracking.movingAverageSpeed;

			file.timeRemaining = tracking.timeRemaining;

			file.timeElapsed = tracking.timeElapsed;

			file.percentUploaded = tracking.percentUploaded;

			file.sizeUploaded = tracking.bytesUploaded;



		} else {

			file.currentSpeed = 0;

			file.averageSpeed = 0;

			file.movingAverageSpeed = 0;

			file.timeRemaining = 0;

			file.timeElapsed = 0;

			file.percentUploaded = 0;

			file.sizeUploaded = 0;

		}

		

		return file;

	};

	

	// Private: Updates the speed tracking object, or creates it if necessary

	SWFUpload.prototype.updateTracking = function (file, bytesUploaded) {

		var tracking = this.fileSpeedStats[file.id];

		if (!tracking) {

			this.fileSpeedStats[file.id] = tracking = {};

		}

		

		// Sanity check inputs

		bytesUploaded = bytesUploaded || tracking.bytesUploaded || 0;

		if (bytesUploaded < 0) {

			bytesUploaded = 0;

		}

		if (bytesUploaded > file.size) {

			bytesUploaded = file.size;

		}

		

		var tickTime = (new Date()).getTime();

		if (!tracking.startTime) {

			tracking.startTime = (new Date()).getTime();

			tracking.lastTime = tracking.startTime;

			tracking.currentSpeed = 0;

			tracking.averageSpeed = 0;

			tracking.movingAverageSpeed = 0;

			tracking.movingAverageHistory = [];

			tracking.timeRemaining = 0;

			tracking.timeElapsed = 0;

			tracking.percentUploaded = bytesUploaded / file.size;

			tracking.bytesUploaded = bytesUploaded;

		} else if (tracking.startTime > tickTime) {

			this.debug("When backwards in time");

		} else {

			// Get time and deltas

			var now = (new Date()).getTime();

			var lastTime = tracking.lastTime;

			var deltaTime = now - lastTime;

			var deltaBytes = bytesUploaded - tracking.bytesUploaded;

			

			if (deltaBytes === 0 || deltaTime === 0) {

				return tracking;

			}

			

			// Update tracking object

			tracking.lastTime = now;

			tracking.bytesUploaded = bytesUploaded;

			

			// Calculate speeds

			tracking.currentSpeed = (deltaBytes * 8 ) / (deltaTime / 1000);

			tracking.averageSpeed = (tracking.bytesUploaded * 8) / ((now - tracking.startTime) / 1000);



			// Calculate moving average

			tracking.movingAverageHistory.push(tracking.currentSpeed);

			if (tracking.movingAverageHistory.length > this.settings.moving_average_history_size) {

				tracking.movingAverageHistory.shift();

			}

			

			tracking.movingAverageSpeed = SWFUpload.speed.calculateMovingAverage(tracking.movingAverageHistory);

			

			// Update times

			tracking.timeRemaining = (file.size - tracking.bytesUploaded) * 8 / tracking.movingAverageSpeed;

			tracking.timeElapsed = (now - tracking.startTime) / 1000;

			

			// Update percent

			tracking.percentUploaded = (tracking.bytesUploaded / file.size * 100);

		}

		

		return tracking;

	};

	SWFUpload.speed.removeTracking = function (file, trackingList) {

		try {

			trackingList[file.id] = null;

			delete trackingList[file.id];

		} catch (ex) {

		}

	};

	

	SWFUpload.speed.formatUnits = function (baseNumber, unitDivisors, unitLabels, singleFractional) {

		var i, unit, unitDivisor, unitLabel;



		if (baseNumber === 0) {

			return "0 " + unitLabels[unitLabels.length - 1];

		}

		

		if (singleFractional) {

			unit = baseNumber;

			unitLabel = unitLabels.length >= unitDivisors.length ? unitLabels[unitDivisors.length - 1] : "";

			for (i = 0; i < unitDivisors.length; i++) {

				if (baseNumber >= unitDivisors[i]) {

					unit = (baseNumber / unitDivisors[i]).toFixed(2);

					unitLabel = unitLabels.length >= i ? " " + unitLabels[i] : "";

					break;

				}

			}

			

			return unit + unitLabel;

		} else {

			var formattedStrings = [];

			var remainder = baseNumber;

			

			for (i = 0; i < unitDivisors.length; i++) {

				unitDivisor = unitDivisors[i];

				unitLabel = unitLabels.length > i ? " " + unitLabels[i] : "";

				

				unit = remainder / unitDivisor;

				if (i < unitDivisors.length -1) {

					unit = Math.floor(unit);

				} else {

					unit = unit.toFixed(2);

				}

				if (unit > 0) {

					remainder = remainder % unitDivisor;

					

					formattedStrings.push(unit + unitLabel);

				}

			}

			

			return formattedStrings.join(" ");

		}

	};

	

	SWFUpload.speed.formatBPS = function (baseNumber) {

		var bpsUnits = [1073741824, 1048576, 1024, 1], bpsUnitLabels = ["Gbps", "Mbps", "Kbps", "bps"];

		return SWFUpload.speed.formatUnits(baseNumber, bpsUnits, bpsUnitLabels, true);

	

	};

	SWFUpload.speed.formatTime = function (baseNumber) {

		var timeUnits = [86400, 3600, 60, 1], timeUnitLabels = ["d", "h", "m", "s"];

		return SWFUpload.speed.formatUnits(baseNumber, timeUnits, timeUnitLabels, false);

	

	};

	SWFUpload.speed.formatBytes = function (baseNumber) {

		var sizeUnits = [1073741824, 1048576, 1024, 1], sizeUnitLabels = ["GB", "MB", "KB", "bytes"];

		return SWFUpload.speed.formatUnits(baseNumber, sizeUnits, sizeUnitLabels, true);

	

	};

	SWFUpload.speed.formatPercent = function (baseNumber) {

		return baseNumber.toFixed(2) + " %";

	};

	

	SWFUpload.speed.calculateMovingAverage = function (history) {

		var vals = [], size, sum = 0.0, mean = 0.0, varianceTemp = 0.0, variance = 0.0, standardDev = 0.0;

		var i;

		var mSum = 0, mCount = 0;

		

		size = history.length;

		

		// Check for sufficient data

		if (size >= 8) {

			// Clone the array and Calculate sum of the values 

			for (i = 0; i < size; i++) {

				vals[i] = history[i];

				sum += vals[i];

			}



			mean = sum / size;



			// Calculate variance for the set

			for (i = 0; i < size; i++) {

				varianceTemp += Math.pow((vals[i] - mean), 2);

			}



			variance = varianceTemp / size;

			standardDev = Math.sqrt(variance);

			

			//Standardize the Data

			for (i = 0; i < size; i++) {

				vals[i] = (vals[i] - mean) / standardDev;

			}



			// Calculate the average excluding outliers

			var deviationRange = 2.0;

			for (i = 0; i < size; i++) {

				

				if (vals[i] <= deviationRange && vals[i] >= -deviationRange) {

					mCount++;

					mSum += history[i];

				}

			}

			

		} else {

			// Calculate the average (not enough data points to remove outliers)

			mCount = size;

			for (i = 0; i < size; i++) {

				mSum += history[i];

			}

		}



		return mSum / mCount;

	};

	

}