package org.weixvn.wae.support;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.weixvn.wae.manager.EngineManager;
import org.weixvn.wae.support.catchbug.FrameDatabaseHelper;

import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager.NameNotFoundException;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.PixelFormat;
import android.graphics.drawable.Drawable;

/**
 * WAE支持类。
 * 
 * <pre>
 * 此类为开发者进行校务系统开发提供了一些基础服务。<br>
 * 如：调用SDK提供的登陆界面、调用SDK提供的议建反馈界面、获登陆验证码、获取已存储的用户帐号和密码。
 * 
 * @author weixvn
 * 
 */
public class WAESupport {

	/**
	 * 验证码图片
	 */
	private Drawable vericode = null;
	/**
	 * 取得登陆信息监听器 <br>
	 * 当开发者向框架程序请求必要的登陆信息（如：帐号、密码、验证码）后，一旦框架向开发者返回数据时就触发此监听器
	 */
	private OnReceiveLoginInfoListenter receiveLoginInfoListener = null;
	/**
	 * 接受广播的对象，在OnReceive()方法中触发各监听器。<br>
	 * 当主框架程序向插件程序返回数据时，主框架会发送相应广播（携带有相应数据）。
	 */
	private ReceiveLoginInfoBroadcastReceiver logininfoBroadcastReceiver = null;

	/**
	 * 调用主框架程序登陆界面的插件程序传入的标题
	 */
	private String title = null;

	/**
	 * 后台任务管理器，用于管理各请求的发送。
	 */
	private WAEUpdateTask updateTask = null;

	/**
	 * 标志后台任务是否取消
	 */
	public boolean isTaskCancelled = false;

	/**
	 * 构造函数 初始化一些必要数据
	 */
	public WAESupport() {
		logininfoBroadcastReceiver = new ReceiveLoginInfoBroadcastReceiver();
		IntentFilter filter = new IntentFilter();
		filter.addAction("LOGINISCLICKED");
		EngineManager.getInstance().getContext()
				.registerReceiver(logininfoBroadcastReceiver, filter);
	}

	/**
	 * 调用框架提供的登陆界面。 <br>
	 * 提供验证码支持
	 * 
	 * @param context
	 *            开发者的应用程序的上下文环境
	 * @param veriCodUrl
	 *            验证码图片URL
	 * @param title
	 *            开发者希望在登陆界面显示的简短的登陆信息描述
	 * 
	 */
	public void launchLoginUi(Context context, String veriCodUrl, String title,
			int system_id) throws Exception {
		Drawable veriCode = null;
		if (veriCodUrl != null && !veriCodUrl.equals("")) {
			getVerificationCode(veriCodUrl);
			while (veriCode == null) {
				Thread.sleep(200);
				veriCode = this.vericode;
			}
		}

		ComponentName com = new ComponentName("org.weixvn.frame",
				"org.weixvn.frame.LoginActivity");
		Intent intent = new Intent();
		Bitmap veriCodeBitmap = null;
		if (veriCode != null) {
			veriCodeBitmap = drawableToBitmap(veriCode);
		}

		this.title = title;
		intent.putExtra("veriCodeBitmap", veriCodeBitmap);
		intent.putExtra("title", title);
		intent.putExtra("systemId", system_id);
		intent.setComponent(com);
		context.startActivity(intent);
	}

	/**
	 * 获取验证码图片。(仅SDK内部使用，开发者无需关心)
	 * 
	 * @param veriCodeUri
	 *            验证码图片的URI
	 */
	protected void getVerificationCode(String veriCodeUri) {

		// 向GetVerificationCode类传递验证码的URL
		EngineManager.getInstance().getWebPageMannger()
				.getWebPage(GetVerificationCode.class)
				.setHtmlValue("veriCodeUri", veriCodeUri);

		// 引起发送Http请求，获取验证码图片，并在GetVerificationCode类的onSuccess()方法
		// 中调用本类的setVerificationCode()方法把图片传递回来。
		EngineManager.getInstance().getWebPageMannger()
				.updateWebPage(GetVerificationCode.class, false);

	}

	/**
	 * 设置验证码图片。开发者无需关心，仅WAE使用。
	 * 
	 * @param vericode
	 */
	public void setVerificationCode(Drawable vericode) {
		this.vericode = vericode;
	}

	/**
	 * 调用框架提供的议建反馈界面
	 * 
	 * @param tile
	 *            议建反馈界面的标题
	 * @param
	 */
	public void launchFeedbackUi(Context context, String title) {
		ComponentName com = new ComponentName("org.weixvn.frame",
				"org.weixvn.frame.FeedbackActivity");

		// 获取插件id
		String pluginId = getPluginId();

		Intent intent = new Intent();

		intent.putExtra("title", title);
		intent.putExtra("Plugin_id", pluginId);
		intent.setComponent(com);
		context.startActivity(intent);

	}

	/**
	 * 把Drawable转换成Bitmap
	 * 
	 * @param drawable
	 * @return Bitmap
	 */
	protected static Bitmap drawableToBitmap(Drawable drawable) {

		Bitmap bitmap = Bitmap
				.createBitmap(
						drawable.getIntrinsicWidth(),
						drawable.getIntrinsicHeight(),
						drawable.getOpacity() != PixelFormat.OPAQUE ? Bitmap.Config.ARGB_8888
								: Bitmap.Config.RGB_565);
		Canvas canvas = new Canvas(bitmap);
		// canvas.setBitmap(bitmap);
		drawable.setBounds(0, 0, drawable.getIntrinsicWidth(),
				drawable.getIntrinsicHeight());
		drawable.draw(canvas);
		return bitmap;
	}

	/**
	 * 设置登陆信息接收监听器。 <br>
	 * 传入一个实现{@link OnReceiveLoginInfoListenter}接口的对象。
	 * 
	 * @param listener
	 *            {@link OnReceiveLoginInfoListenter} 类型
	 */
	public void setOnReceiveLoginInfoListener(
			OnReceiveLoginInfoListenter listener) {
		this.receiveLoginInfoListener = listener;
	}

	/**
	 * 触发任务更新的执行
	 * 
	 * @param updateTask
	 *            WAEUpdateTask的子类对象， 注意：每调用一次update方法，传入的updateTask参数都是不同的。
	 * @param context
	 *            应用程序上下文，建议传入Activity.this
	 * @param title
	 *            进度条的标题
	 * @param content
	 *            进度条的内容
	 * @param max
	 *            任务的数量，可以是发送请求的次数
	 * @param params
	 *            The parameters of the task.将被传入doInBackground(Integer...
	 *            params)方法中
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	public void update(WAEUpdateTask updateTask, Context context, String title,
			String content, int max, Integer... params)
			throws InstantiationException, IllegalAccessException {
		this.updateTask = updateTask;
		this.updateTask.initProgressDialog(context, title, content, max);
		this.isTaskCancelled = false;
		this.updateTask.execute(params);
	}

	/**
	 * 框架返回用户登陆信息。由框架程序调用，开发者无需关心。
	 * 
	 * @param info
	 *            用户输入的登陆信息，如：用户名、密码、验证码。
	 */
	public void receiveLoginInfo(Map<String, String> info) {

		// 触发监听器响应，让开发者接收到主框架返回的信息
		this.receiveLoginInfoListener.onReceiveLoginInfo(info);
	}

	/**
	 * 登陆信息广播接收器。 <br>
	 * 主框架程序以广播的形式返回用户输入的用户名、密码等信息。此接收器即是用来接收广播并将接收到的数据返回给开发者。
	 * 
	 * @author weixvn
	 * 
	 */
	protected class ReceiveLoginInfoBroadcastReceiver extends BroadcastReceiver {

		@Override
		public void onReceive(Context context, Intent intent) {
			if (intent.getAction().equals("LOGINISCLICKED")) {
				String id = intent.getStringExtra("id");
				String password = intent.getStringExtra("password");
				String veriCode = intent.getStringExtra("veriCode");
				String title = intent.getStringExtra("title");
				Map<String, String> info = new HashMap<String, String>();
				info.put("id", id);
				info.put("password", password);
				info.put("veriCode", veriCode);

				// 触发监听器，将接收到的数据返回给开发者
				// 判断输入
				if (WAESupport.this.title != null
						&& WAESupport.this.title.equals(title)) {
					receiveLoginInfo(info);
				}
				// 取消广播监听
				// EngineManager.getInstance().getContext().unregisterReceiver(this);
			}
		}

	}

	/**
	 * 获得单个学校系统账户和密码
	 * 
	 * <pre>
	 * 当用户首次登陆或数据库中已不存在用户的帐号信息或登陆需要验证码时，会弹出登陆界面。
	 * 当帐号登陆成功后，再次登陆时，不会弹出登陆界面。<br>
	 * 要想强制控制登陆界面是否弹出，请参看同名方法{@link getAccountPassword(int System_id, Context context,String veriCodeUrl,String title,boolean flag)}
	 * 
	 * @param System_id
	 *            学校系统Id
	 * @param context
	 *            开发者的应用程序的上下文环境
	 * @param veriCodeUrl	验证码Url
	 * @param title		标题
	 * @return 返回一个map，包含系统账户（键：account）和密码（键：password）。
	 * @throws Exception
	 */
	public void getAccountPassword(int System_id, Context context,
			String veriCodeUrl, String title) throws Exception {
		// getAccountPassword(System_id, context, veriCodeUrl, title, false);
		// 如果有验证码，直接调用登陆界面，并返回
		if ((veriCodeUrl != null && !"".equals(veriCodeUrl))) {
			launchLoginUi(context, veriCodeUrl, title, System_id);
			return;
		}

		SQLiteDatabase db;
		Map<String, String> accountPassword = null;
		db = getFrameDB(context);
		String selectString = "SELECT * FROM UserSystem WHERE System_id="
				+ System_id;
		Cursor cursor = db.rawQuery(selectString, null);
		if (cursor.getCount() > 0) {
			accountPassword = new HashMap<String, String>();
			cursor.moveToFirst();
			accountPassword.put("id", cursor.getString(cursor
					.getColumnIndex("UserSystem_username")));
			accountPassword.put("password", cursor.getString(cursor
					.getColumnIndex("UserSystem_userpassword")));
		}
		cursor.close();
		db.close();
		if (accountPassword != null) {
			// 如果从框架主程序数据库中查到帐户密码　则触发监听器响应，让开发者接收到主框架返回的信息
			this.receiveLoginInfoListener.onReceiveLoginInfo(accountPassword);
		} else { // 从框架主程序数据库中找不到帐户密码，则调出登陆界面
			launchLoginUi(context, veriCodeUrl, title, System_id);
		}
	}

	/**
	 * 获得单个学校系统账户和密码
	 * 
	 * @param System_id
	 *            学校系统Id
	 * @param context
	 *            开发者的应用程序的上下文环境
	 * @param veriCodeUrl
	 *            验证码Url
	 * @param title
	 *            标题
	 * @param flag
	 *            是否强制弹出登陆界面(true:一定弹出登陆界面.)。
	 * @return 返回一个map，包含系统账户（键：account）和密码（键：password）。
	 * @throws Exception
	 */
	public void getAccountPassword(int System_id, Context context,
			String veriCodeUrl, String title, boolean flag) throws Exception {

		// 如果有验证码，直接调用登陆界面，并返回
		if (flag || (veriCodeUrl != null && !"".equals(veriCodeUrl))) {
			launchLoginUi(context, veriCodeUrl, title, System_id);
			return;
		}

		SQLiteDatabase db;
		Map<String, String> accountPassword = null;
		db = getFrameDB(context);
		String selectString = "SELECT * FROM UserSystem WHERE System_id="
				+ System_id;
		Cursor cursor = db.rawQuery(selectString, null);
		if (cursor.getCount() > 0) {
			accountPassword = new HashMap<String, String>();
			cursor.moveToFirst();
			accountPassword.put("id", cursor.getString(cursor
					.getColumnIndex("UserSystem_username")));
			accountPassword.put("password", cursor.getString(cursor
					.getColumnIndex("UserSystem_userpassword")));
		}
		cursor.close();
		db.close();
		// flag == false，不调用登陆界面
		if (flag == false) {
			if (accountPassword != null) {
				this.receiveLoginInfoListener
						.onReceiveLoginInfo(accountPassword);
			}
			return;
		}
		if (accountPassword != null) {
			// 如果从框架主程序数据库中查到帐户密码　则触发监听器响应，让开发者接收到主框架返回的信息
			this.receiveLoginInfoListener.onReceiveLoginInfo(accountPassword);
		} else { // 从框架主程序数据库中找不到帐户密码，则调出登陆界面
			launchLoginUi(context, veriCodeUrl, title, System_id);
		}
	}

	/**
	 * 获得单个学校系统账户和密码
	 * 
	 * <pre>
	 * 直接以Map形式返回数据 
	 * @param System_id
	 *            学校系统Id
	 * @param context
	 *            开发者的应用程序的上下文环境
	 * @return 返回一个map，包含系统账户（键：id）和密码（键：password）。
	 * @throws Exception
	 */
	public Map<String, String> getAccountPassword(int System_id, Context context)
			throws Exception {

		SQLiteDatabase db;
		Map<String, String> accountPassword = null;

		db = getFrameDB(context);

		String selectString = "SELECT * FROM UserSystem WHERE System_id="
				+ System_id;
		Cursor cursor = db.rawQuery(selectString, null);
		if (cursor.getCount() > 0) {
			accountPassword = new HashMap<String, String>();
			cursor.moveToFirst();
			accountPassword.put("id", cursor.getString(cursor
					.getColumnIndex("UserSystem_username")));
			accountPassword.put("password", cursor.getString(cursor
					.getColumnIndex("UserSystem_userpassword")));
		}
		cursor.close();
		db.close();

		return accountPassword;
	}

	/**
	 * 更新框架主程序数据库，更新校园系统id、用户名、密码等
	 * 
	 * @param schoolSystemId
	 *            校园系统id(如：新务处为1，教务处2)
	 * @param account
	 *            学号
	 * @param password
	 *            密码
	 * @param schoolSysName
	 *            校园系统的名字
	 */
	public void updateUserSystemInfo(int schoolSystemId, String account,
			String password, String schoolSysName) {
		title = schoolSysName;
		int userId = 0;
		String xh = null;
		SQLiteDatabase db;
		try {
			db = getFrameDB(EngineManager.getInstance().getContext());
			Cursor cursor = db.rawQuery("SELECT * FROM User_info", null);

			int recordSum = cursor.getCount();
			if (recordSum > 0) { // 非匿名登陆
				cursor.moveToFirst();
				userId = cursor.getInt(cursor.getColumnIndex("User_id"));
				xh = cursor.getString(cursor.getColumnIndex("User_num"));
			}

			// 匿名登陆或输入的学号与主程序不同，直接返回
			if (recordSum < 0 || xh == null || !xh.equals(account)) {
				cursor.close();
				db.close();
				return;
			}
			cursor.close();
			db.close();
			updateUserSystemInfo(EngineManager.getInstance().getContext(),
					userId, schoolSystemId, account, password, ""
							+ schoolSystemId);
		} catch (NameNotFoundException e1) {
			e1.printStackTrace();
		}
	}

	/**
	 * 更新框架主程序数据库，更新校园系统id、用户名、密码等
	 * 
	 * @param context
	 *            程序上下文环境
	 * @param userId
	 *            用户id(数据库主键，不时用户名)
	 * @param systemId
	 *            校园系统id
	 * @param account
	 *            校园系统用户名
	 * @param password
	 *            校园系统密码
	 * @param UserSystem_mark
	 *            保留字段
	 * @throws NameNotFoundException
	 */
	public void updateUserSystemInfo(Context context, int userId, int systemId,
			String account, String password, String UserSystem_mark)
			throws NameNotFoundException {
		SQLiteDatabase db = getFrameDB(context);
		String selectString = "SELECT * FROM UserSystem WHERE System_id="
				+ systemId;
		Cursor cursor = db.rawQuery(selectString, null);

		if (cursor.getCount() > 0) {
			cursor.moveToFirst();
			for (int i = 0; i < cursor.getCount(); i++) {

				if (cursor.getInt(cursor.getColumnIndex("System_id")) == systemId) {
					String updateString = "update UserSystem set User_id = "
							+ userId + "," + "System_id=" + systemId + ","
							+ "UserSystem_name='" + title + "',"
							+ " UserSystem_username= '" + account
							+ "',UserSystem_userpassword='" + password
							+ "'WHERE System_id=" + systemId + ";";
					db.execSQL(updateString);
					break;
				}
				cursor.moveToNext();
			}

		} else {
			// String deleteStr = "delete from UserSystem where System_id="
			// + systemId + ";";
			// db.execSQL(deleteStr);
			String insertString = "insert into UserSystem(User_id,System_id,UserSystem_name,UserSystem_username,UserSystem_userpassword)"
					+ " values ("
					+ userId
					+ ","
					+ systemId
					+ ",'"
					+ title
					+ "','" + account + "','" + password + "');";
			db.execSQL(insertString);
			cursor.close();
			db.close();
		}

	}

	/**
	 * 获取代理
	 * 
	 * @param System_id
	 *            校园系统id
	 * @param context
	 *            程序上下文环境
	 * @return
	 * @throws NameNotFoundException
	 */
	public List<String> getProxy() throws NameNotFoundException {

		List<String> proxys = null;
		SQLiteDatabase db = getFrameDB(EngineManager.getInstance().getContext());
		String selectProxys = "select Proxy_domain,Proxy_port from Proxy_info";
		Cursor cursor = db.rawQuery(selectProxys, null);
		if (cursor == null) {
			return null;
		}
		if (cursor.getCount() != 0) {
			cursor.moveToFirst();

			proxys = new ArrayList<String>();
			while (!cursor.isAfterLast()) {
				String domain = cursor.getString(cursor
						.getColumnIndex("Proxy_domain"));
				String port = cursor.getString(cursor
						.getColumnIndex("Proxy_port"));

				proxys.add(domain + ":" + port);
				cursor.moveToNext();
			}
		}
		cursor.close();
		db.close();
		return proxys;
	}

	/**
	 * 获取微讯平台帐户密码
	 * 
	 * @param context
	 *            应用程序上下文
	 * @return　Map<String, String>平台帐户、密码，对应的键分别为nickname、password
	 * @throws NameNotFoundException
	 */
	public Map<String, String> getPlatformAccountPassword(Context context)
			throws NameNotFoundException {
		Map<String, String> PlatformAccountPassword = null;
		SQLiteDatabase db = getFrameDB(context);
		Cursor cursor = db.query("User_info", new String[] { "User_nickname",
				"User_password" }, null, null, null, null, null);
		if (cursor.getCount() != 0) {
			cursor.moveToFirst();
			String User_nickname = cursor.getString(cursor
					.getColumnIndex("User_nickname"));
			String User_password = cursor.getString(cursor
					.getColumnIndex("User_password"));
			PlatformAccountPassword = new HashMap<String, String>();
			PlatformAccountPassword.put("nickname", User_nickname);
			PlatformAccountPassword.put("password", User_password);
		}
		db.close();
		return PlatformAccountPassword;
	}

	/**
	 * 获取主框架程序的数据库
	 * 
	 * @param context
	 *            应用程序上下文
	 * @return
	 * @throws NameNotFoundException
	 */
	private SQLiteDatabase getFrameDB(Context context)
			throws NameNotFoundException {
		Context packageContext = context.createPackageContext(
				"org.weixvn.frame", 2);
		FrameDatabaseHelper helper = new FrameDatabaseHelper(packageContext,
				"frame_db", 1);
		SQLiteDatabase db = helper.getWritableDatabase();
		return db;
	}

	/**
	 * 解析assets/wae/plugin.xml文件
	 * 
	 * @return
	 */
	public Document parseWAEPluginXML() {
		Document doc = null;
		try {
			InputStream is = EngineManager.getInstance().getContext()
					.getApplicationContext().getAssets().open("wae/plugin.xml");

			if (is != null) {
				doc = Jsoup.parse(is, "utf-8", "");
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		return doc;
	}

	/**
	 * 获取插件id(写在wae/plugin.xml的PluginID标签下)
	 * 
	 * @return 插件id
	 */
	public String getPluginId() {
		Document doc = parseWAEPluginXML();
		String pluginId = null;
		try {
			pluginId = doc.getElementsByTag("PluginID").first().text();
		} catch (NullPointerException e) {
			e.printStackTrace();
		}
		return pluginId;
	}

}
