package org.weixvn.wae.webpage;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.weixvn.wae.manager.EngineManager;
import org.weixvn.wae.storage.DatabaseHelper;
import org.weixvn.wae.webpage.net.WebPageHandler;

import com.loopj.android.http.RequestParams;

/**
 * 网页基类
 * <p>
 * 每个要解析的网页可从此类派生，并且实现两个抽象函数 {@link #onStart()}和 {@link #onSuccess(InputStream)}
 * 两个函数。需要的时候可以实现{@link #onFailure(Throwable)}等函数，以处理网络请求的时候可能出现的其它异常。
 * 
 * @author weixvn
 */
public abstract class WebPage extends WebPageHandler {
	private Map<String, String> htmlArray = null; // 网页数据集
	protected WebPageMannger wfm; // page 管理器
	protected DatabaseHelper db; // page 数据库
	protected String charsetName = null; // 网页解析字符编码,默认为“UTF-8”。如果设置为null，将由网页的http-equiv
											// meta标签决定。
	protected String baseUri = ""; // 网页来源的URL 默认为空字符串

	/**
	 * 网页请求参数表 {@link RequestParams}
	 */
	protected RequestParams params;
	/**
	 * 请求地址,如：http://www.baidu.com
	 */
	protected String uri;

	/**
	 * 请求类型{@link RequestType}
	 */
	protected RequestType type;

	/**
	 * HTTP请求类型的枚举
	 * <p>
	 * 目前支持GET, POST, PUT, DELETE四种网络请求
	 * 
	 * @author weixvn
	 */
	public enum RequestType {
		GET, POST, PUT, DELETE
	}

	/**
	 * 构造方法，实现一些初始化工作。
	 */
	public WebPage() {
		this.wfm = EngineManager.getInstance().getWebPageMannger();
		this.htmlArray = new HashMap<String, String>();
		this.db = EngineManager.getInstance().getDB();
	}

	/**
	 * 查询网页反馈的数据。 <br>
	 * 开发者要首先调用{@link #setHtmlValue(String key, String value)}方法设置好键和值
	 * ，用此方法取得键key对应的值。
	 * 
	 * @param key
	 *            标签关键字
	 * @return key对应的值，若不存在则返回null.
	 */
	public String getHtmlValue(String key) {
		if (htmlArray.containsKey(key))
			return (String) htmlArray.get(key);

		return null;
	}

	/**
	 * 设置网页数据 <br>
	 * 此方法存储的值，类似于键－值对。键和值一一对应，可据键来调用{@link #getHtmlValue(String key)}而得到相应的值。
	 * 
	 * @param key
	 *            标签关键字
	 * @param value
	 *            标签数据
	 */
	public void setHtmlValue(String key, String value) {
		htmlArray.put(key, value);
	}

	/**
	 * 获取网页请求参数表
	 * 
	 * @return RequestParams 网页请求参数表
	 */
	public RequestParams getParams() {
		if (params == null) {
			params = new RequestParams();
		}
		return params;
	}

	/**
	 * 设置网页请求参数表
	 * 
	 * @param params
	 *            请求参数表
	 */
	public void setParams(RequestParams params) {
		this.params = params;
	}

	/**
	 * 设置请求的URL
	 * 
	 * @param uri
	 *            请求的url
	 */
	public void setURI(String uri) {
		this.uri = uri;
	}

	/**
	 * 获取请求的url
	 * 
	 * @return 请求的url，可能为空
	 */
	public String getURI() {
		return this.uri;
	}

	/**
	 * 设置请求类型
	 * 
	 * @param type
	 *            请求类型{@link RequestType}
	 */
	public void setType(RequestType type) {
		this.type = type;
	}

	/**
	 * 获取请求类型
	 * 
	 * @return RequestType 请求类型{@link RequestType}
	 */
	public RequestType getType() {
		return type;
	}

	/**
	 * 网页请求开始
	 * 
	 * <pre>
	 * 网页请求一般需要完成以下设置：
	 * 请求地址 {@link #uri}
	 * 请求参数 {@link #params}
	 * 请求类型
	 * {@link #type}
	 * 
	 * <pre>
	 */
	@Override
	abstract public void onStart();

	/**
	 * 网页请求成功
	 * 
	 * <pre>
	 * 此方法与{@link #onSuccess(Document html)}方法功能相似。
	 * 当服务器返回的是纯Html脚本时建议使用{@link #onSuccess(Document html)}进行解析，
	 * 当服务器返回的是图片等时建议使用复写{@link #onSuccess(InputStream html)}方法进行解析
	 * <br>
	 * @param html 服务器以InputStream类型返回的网页数据，如：html脚本、图片等。当html时图片时，复写时删除eclipse等发工具自动添加的super.onSuccess();
	 */
	@Override
	public void onSuccess(InputStream html) {
		Document doc = null;
		try {
			doc = Jsoup.parse(html, charsetName, baseUri);
			this.onSuccess(doc);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 网页请求成功 <br>
	 * 服务器返回的网页脚本数据，已转化成Document类型，可以解析此Document获取有用的数据
	 * 
	 * @param html
	 *            返回的Html网页，已转化成Document类型
	 */
	abstract public void onSuccess(Document html);

	/**
	 * 设置网页解析时使用的字符编码。<br>
	 * <br>
	 * 建议在WebPage子类的onStart()方法中调用。
	 * 
	 * @param charsetName
	 *            网页解析字符编码,默认为“UTF-8”。如果设置为null，将由网页的http-equiv meta标签决定。
	 */
	public void setCharsetName(String charsetName) {
		this.charsetName = charsetName;
	}

	/**
	 * 设置网页来源的URL，用来解决相对链接问题。(若没用相对链接问题，可忽略) <br>
	 * 建议在WebPage子类的onStart()方法中调用。
	 * 
	 * @param baseUri
	 *            网页来源的URL 默认为空字符串
	 */
	public void setBaseUri(String baseUri) {
		this.baseUri = baseUri;
	}

	/**
	 * 为HttpClient设置Header
	 * 
	 * @param header
	 *            Header名称，比如：“Referer”
	 * @param value
	 *            Header的值
	 */
	public void addHeader(String header, String value) {
		EngineManager.getInstance().getWebPageMannger().getHttpClient()
				.addHeader(header, value);
	}

	/**
	 * 为HttpClient添加referer
	 * 
	 * @param referer
	 */
	public void addReferer(String referer) {
		this.addHeader("Referer", referer);
	}

}
