package org.weixvn.wae.webpage;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.HashMap;
import java.util.Map;

import org.apache.http.client.CookieStore;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.weixvn.wae.manager.EngineManager;
import org.weixvn.wae.webpage.WebPage.RequestType;
import org.weixvn.wae.webpage.net.SSLSocketFactoryEx;

import android.util.Log;

import com.loopj.android.http.AndroidHttpClient;
import com.loopj.android.http.PersistentCookieStore;

/**
 * 网页管理类，提供网页实例的存储与查询，网页数据更新接口。
 * 
 * @author weixvn
 */
public class WebPageMannger {
	// private Queue<WebPage> webPageQueue;
	private Map<Class<?>, WebPage> webPageArray;
	private AndroidHttpClient httpClient; // HTTP请求客户端
	private CookieStore store;

	/**
	 * 构造网页管理器，做一些网页管理初化工作。
	 */
	public WebPageMannger() {
		this.httpClient = this.getHttpClient();
		webPageArray = new HashMap<Class<?>, WebPage>();
	}

	/**
	 * 获取网页类的实例，使用方式如：getWebPage(WebPage.Class)。
	 * 
	 * @param object
	 *            网页类的Class
	 * @return 函数会返回WebPage子类的实例，不存在时会创建新的实例。
	 */
	public WebPage getWebPage(Class<?> object) {
		if (webPageArray.containsKey(object)) {
			return webPageArray.get(object);
		}

		WebPage page = null;
		try {
			page = (WebPage) object.newInstance();
			if (page != null) {
				webPageArray.put(object, page);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		return page;
	}

	/**
	 * 更新网页数据，并处理请求后的网页
	 * 
	 * @param object
	 *            网页类的Class
	 * @param isAsync
	 *            是否异步请求
	 */
	public void updateWebPage(Class<?> object, boolean isAsync) {

		// 当后台任务被取消时，直接返回
		if (EngineManager.getInstance().getWaeSupport().isTaskCancelled) {
			return;
		}

		WebPage page = this.getWebPage(object);

		// 装载页面
		page.onStart();

		// 地址为空
		String url = page.getURI();
		RequestType type = page.getType();
		if (url == null || type == null) {
			Log.e(getClass().getSimpleName().toString(),
					"One http request,the url is " + url
							+ " \nRequest type is " + type.toString());
			return;
		}

		// 检索网页请求类型
		switch (type) {
		case GET:
			this.httpClient.get(url, page.getParams(), page, isAsync);
			break;
		case POST:
			this.httpClient.post(url, page.getParams(), page, isAsync);
			break;
		case PUT:
			this.httpClient.put(url, page.getParams(), page, isAsync);
			break;
		case DELETE:
			this.httpClient.delete(url, page, isAsync);
			break;
		default:
			return;
		}
	}

	/**
	 * 清理所有的网页
	 */
	public void clearPages() {
		webPageArray.clear();
	}

	/**
	 * 获取HttpClient
	 * 
	 * @return AndroidHttpClient
	 */
	public AndroidHttpClient getHttpClient() {
		if (this.httpClient == null) {
			this.httpClient = new AndroidHttpClient();
			this.httpClient.setCookieStore(getCookieStore());
			this.resetSSLSocketFactory();
		}
		return this.httpClient;
	}

	/**
	 * 获得CookieStore
	 * 
	 * @return CookieStore 默认为空时创建一个新的实例
	 */
	public CookieStore getCookieStore() {
		if (this.store == null)
			this.store = new PersistentCookieStore(EngineManager.getInstance()
					.getContext());
		return this.store;
	}

	/**
	 * 重新设置SSLSocketFactory,使HttpClient忽略安全验证，能够访问所有https://类型的URL
	 */
	public void resetSSLSocketFactory() {
		KeyStore trustStore = null;
		try {
			trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
		} catch (KeyStoreException e) {
			e.printStackTrace();
		}
		try {
			trustStore.load(null, null);
		} catch (NoSuchAlgorithmException e1) {
			e1.printStackTrace();
		} catch (CertificateException e1) {
			e1.printStackTrace();
		} catch (IOException e1) {
			e1.printStackTrace();
		}

		SSLSocketFactoryEx sf = null;
		try {
			sf = new SSLSocketFactoryEx(trustStore);
		} catch (KeyManagementException e) {
			e.printStackTrace();
		} catch (UnrecoverableKeyException e) {
			e.printStackTrace();
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		} catch (KeyStoreException e) {
			e.printStackTrace();
		}
		sf.setHostnameVerifier(SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
		this.httpClient.setSSLSocketFactory(sf);
	}

	/**
	 * 清除Cookie，重置HttpClient，与服务器断开连接。
	 */
	public void resetHttpClient() {
		this.httpClient.resetHttpClient();
	}

}
