<?php 
/*
 * How to use?
 * 
 * 1. Initialize
 * $openApiClient = new OpenApiClient('siteurl','sitekey');
 * 
 * 2. Set format for response, if not, will return default format 'json'.
 * $openApiClient->setFormat('xml'); or  $openApiClient->setFormat('json');
 * 
 * 3. Call API, if API define 'GET' method for http request, 
 *    call 'get' method, or call 'post' method 
 * $result = $openApiClient->get('friend.getfriends', array('uid' => 1));
 * OR
 * $result = $openApiClient->post('thread.post', array('subject' => 'test','content' => 'test'));
 * 
 * 4. Test response
 * var_dump($result);
 *
 */
class OpenApiClient{
	public $timestamp;
	public $format = 'json';
	private $siteUrl;
	private $sitekey;
	
	public function __construct ($siteUrl, $sitekey) {
		$this->siteUrl = rtrim($siteUrl,'/') . '/openapi.php';
		$this->sitekey = $sitekey;
        $this->timestamp = time();
	}
	
	public function setFormat($format){
		if (trim($format) != '' && in_array($format, array('xml','json'))){
			$this->format = $format;
		}
	}
	
	public function get($callback, $args){
		return $this->_invoke($callback,$args,'GET');
	}
	
	public function post($callback,$args){
		return $this->_invoke($callback,$args,'POST');
	}
	
	public function _invoke($callback, $args, $httpMethod = 'GET') {
		$params = array();
		$params['callback'] = $callback;
		$params['format']   = $this->format;
		$params['charset']  = 'utf-8';
		$params['v']        = '1.0';
		$params['params']   = $args ? json_encode($args) : '';

		ksort($params);
		$str = '';
		foreach ($params as $k => $v) {
            if ($v) {
                $str .= $k . '=' . $v . '&';
            }
		}
		$params['sign'] = md5($str . $this->sitekey);
		list($errno, $result) = $this->_postRequest($params, $httpMethod);		
		return !$errno ? $result : false; 
	}
	
    private function _postRequest($params, $httpMethod) {
		$result = '';
		$str = $this->_formatParams($params);
		if (function_exists('curl_init')) {
			$isPost = strtolower($httpMethod) == 'post' ? 1 : 0;
			$ch = curl_init();
			if ($isPost){
				curl_setopt($ch, CURLOPT_URL, $this->siteUrl);
				curl_setopt($ch, CURLOPT_POST, 1);
				curl_setopt($ch, CURLOPT_POSTFIELDS, $str);
			}else{
				curl_setopt($ch, CURLOPT_URL, $this->siteUrl . '?' . $str);
				curl_setopt($ch, CURLOPT_HTTPGET, 1);
			}
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($ch, CURLOPT_USERAGENT, 'PHPWind OPENAPI 1.0 (curl) ' . phpversion());
			$result = curl_exec($ch);
			$errno = curl_errno($ch);
			curl_close($ch);
			
			return array($errno, $result);
		} else {
			$context = array(
				'http' => array(
					'method' => $httpMethod,
					'header' => 'Content-type: application/x-www-form-urlencoded'."\r\n" . 
								'User-Agent: PHPWind OPENAPI 1.0 (non-curl) ' . phpversion() . "\r\n".
								'Content-length: ' . strlen($str),
					'content' => $str
				)
			);
			$url = $httpMethod == 'GET' ? $this->siteUrl . '?' . $str : $this->siteUrl;
			$contextid = stream_context_create($context);
			$sock = fopen($url, 'r', false, $contextid);
			if ($sock) {
				while (!feof($sock)) {
					$result .= fgets($sock, 4096);
				}
				fclose($sock);
			}
		}

		return array(0, $result);
	}
	
	private function _formatParams($params) {
		$str = '';
		foreach ($params as $k => $v) {
			if (is_array($v)) {
				foreach ($v as $kv => $vv) {
					$str .= '&' . $k . '[' . $kv  . ']=' . urlencode($vv);
				}
			} else {
				$str .= '&' . $k . '=' . urlencode($v);
			}
		}
		return ltrim($str,'&');
	}
}