<?php
!defined('M_P') && exit('Forbidden');
/**
 * 商品服务层
 * @author L.IuHu.I@2010/8/24
 *
 */
class PW_ProductsService {
	
	/**
	 * 商品DAO服务对象
	 * @access private
	 * @var PW_Dianpu_ProductsDB
	 */
	var $_productsDao;
	
	/**
	 * 构造函数，（兼容PHP4）
	 */
	function PW_ProductsService() {
		$this->__construct();
	}
	
	/**
	 * 构造函数,(兼容PHP5)
	 */
	function __construct() {
		$this->_productsDao = $this->_getProductsDao();
	}
	
	/**
	 * 新增一个商品
	 * @param array $fieldData array('字段' => '值')需要添加的商品信息
	 * @return mixed boolean (false:操作失败) | integer 返回插入的ID
	 */
	function addProduct($fieldData) {
		$fieldData = $this->_checkProduct($fieldData);
		if (!$fieldData) return false;
		$fieldData['createdtime'] = $fieldData['modifiedtime'] = $GLOBALS['timestamp'];
		return $this->_productsDao->insert($fieldData);
	}
	
	/**
	 * 根据商品ID获取一个商品
	 * @param integer $productId 商品ID
	 * @return mixed boolean (false:操作失败) | array array('字段' => '值')返回查询得到的商品信息
	 */
	function getProduct($productId) {
		$productId = intval($productId);
		if (!$productId) return false;
		return $this->_productsDao->get($productId);
	}
	
	/**
	 * 通过商品ID列表获得商品信息列表
	 * @param array $productIds 商品ID列表
	 * @param boolean $order 是否排序
	 * @return mixed boolean (false:操作失败) | array array(array('字段' => '值'),...)返回商品信息列表
	 */
	function getProductsByProductIds($productIds, $order = false) {
		if (!S::isArray($productIds)) return false;
		$result = $this->_productsDao->getsByIds($productIds);
		if ($order) return $this->_orderResult($productIds, $this->_bindPhotos($result));
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 根据商品ID和店铺ID删除一个商品
	 * @param integer $productId 商品ID
	 * @param integer $objectId 商品所在的店铺ID
	 * @return mixed boolean (false:操作失败) | integer 返回受影响的行数
	 */
	function deleteProduct($productId, $objectId) {
		$objectId = intval($objectId);
		$productId = intval($productId);
		if (!$productId || !$objectId) return false;
		return $this->_productsDao->delete($productId, $objectId);
	}
	
	/**
	 * 更新一个商品
	 * @param array $fieldData 待更新的商品信息
	 * @param integer $productId 待更新的商品ID
	 * @param integer $objectId 待更新的商品所在的店铺ID
	 * @return mixed boolean (false:操作失败) | integer 返回受影响的行数
	 */
	function updateProduct($fieldData, $productId, $objectId) {
		$fieldData = $this->_checkProduct($fieldData);
		$objectId = intval($objectId);
		$productId = intval($productId);
		if (!$productId || !S::isArray($fieldData) || !$objectId) return false;
		$fieldData['modifiedtime'] = $GLOBALS['timestamp'];
		return $this->_productsDao->update($fieldData, $productId, $objectId);
	}
	
	/**
	 * 获取所有商品
	 * @param integer $page 开始搜索的页码
	 * @param integer $perpage 搜索的条数
	 * @return array array(array('字段' => '值'),...) 返回组装有封面图片的商品信息列表
	 */
	function getProducts($page, $perpage) {
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->gets($start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 统计所有商品
	 * @return integer 返回统计结果
	 */
	function countProducts() {
		return $this->_productsDao->count();
	}
	
	/**
	 * 获取店铺商品
	 * @param integer $objectid 需要获取的商品所在的店铺ID
	 * @param integer $page 需要获取的商品列表的页码位置
	 * @param integer $perpage 需要获取的商品列表显示的条数
	 * @return mixed boolean (false:操作失败) | array array(array('字段' => '值'),...) 返回商品信息列表
	 */
	function getProductsByObjectId($objectid, $page, $perpage) {
		$objectid = intval($objectid);
		if (!$objectid) return false;
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->getsByObjectId($objectid, $start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 统计店铺商品
	 * @param integer $objectid 需要统计商品总数的店铺ID
	 * @return mixed boolean (false:操作失败) | integer 返回统计的结果
	 */
	function countProductsByObjectId($objectid) {
		$objectid = intval($objectid);
		if (!$objectid) return false;
		return $this->_productsDao->countByObjectId($objectid);
	}
	
	/**
	 * 获取店铺推荐商品
	 * @param integer $objectid 需要获取的商品所在的店铺ID
	 * @param integer $page 需要获取的商品列表的页码位置
	 * @param integer $perpage 需要获取的商品列表显示的条数
	 * @return mixed boolean (false:操作失败) | array array(array('字段' => '值'),...) 返回商品信息列表
	 */
	function getProductsByObjectIdWithRecommend($objectid, $page, $perpage) {
		$objectid = intval($objectid);
		if (!$objectid) return false;
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->getsByObjectIdWithRecommend($objectid, $start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 统计店铺推荐商品
	 * @param integer $objectid 需要统计的店铺ID
	 * @return mixed boolean (false:操作失败) | integer 返回统计的结果
	 */
	function countProductsByObjectIdWithRecommend($objectid) {
		$objectid = intval($objectid);
		if (!$objectid) return false;
		return $this->_productsDao->countByObjectIdWithRecommend($objectid);
	}
	
	/**
	 * 获得所有推荐商品分页列表
	 * @param integer $page 查找的分页
	 * @param integer $perpage 查找的条数
	 * @return array array(array('字段' => '值'),...) 返回含有封面图片的商品列表
	 */
	function getProductsWithRecommend($page, $perpage) {
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->getsWithRecommend($start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 统计所有推荐商品
	 * @return integer 返回统计结果
	 */
	function countProductsWithRecommend() {
		return $this->_productsDao->countWithRecommend();
	}
	
	/**
	 * 根据商品点击率获取商品
	 * @param integer $page 查询的分页
	 * @param integer $perpage 查询的条数
	 * @return array array(array('字段' => '值'),...) 返回商品信息列表
	 */
	function getProductsWithHits($page, $perpage) {
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$products = $this->_productsDao->getsByProductWithHits($start, $perpage);
		return $this->_bindPhotos($products);
	}
	
	/**
	 * 获取店铺已上架商品列表
	 * @param integer $objectid 需要获取商品所在的店铺ID
	 * @param integer $page 需要获取的查询页数
	 * @param integer $perpage 需要获取的条数
	 * @return mixed boolean (false：操作失败) | array array(array('字段' => '值'),...) 返回商品信息列表
	 */
	function getsByObjectIdWithShelf($objectid, $page, $perpage) {
		$objectid = intval($objectid);
		if (!$objectid) return false;
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->getsByObjectIdWithShelf($objectid, $start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 统计店铺已上架商品
	 * @param integer $objectid 需要统计的店铺ID
	 * @return mixed boolean (false:操作失败) | integer 返回统计结果
	 */
	function countByObjectIdWithShelf($objectid) {
		$objectid = intval($objectid);
		if (!$objectid) return false;
		return $this->_productsDao->countByObjectIdWithShelf($objectid);
	}
	
	/**
	 * 根椐店铺IDs统计商品个数
	 * @param array $objectIds 需要统计的店铺ID列表
	 * @return mixed boolean (false:操作失败) | integer 统计总数
	 */
	function countByObjectIds($objectIds) {
		return $this->_productsDao->countByObjectIds($objectIds);
	}
	
	/**
	 * 根椐店铺IDs获取商品
	 * @param array $objectIds 店铺ID列表
	 * @param integer $page 开始获取的页数
	 * @param integer $perpage  获得的条数
	 * @return array array(array('字段' => '值'),...) 返回数组信息
	 */
	function getsByObjectIds($objectIds, $page, $perpage) {
		list($start, $offset) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->getsByObjectIds($objectIds, $start, $offset);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 根据商品ID和店铺ID删除多个商品
	 * @param array $productIds 需要删除的商品ID列表
	 * @param integer $objectId 店铺ID
	 * @return mixed boolean (false:操作失败) | integer 返回受影响的行数
	 */
	function deleteProducts($productIds, $objectId) {
		if (!$productIds || !$objectId) return false;
		return $this->_productsDao->deletes($productIds, $objectId);
	}
	
	/**
	 * 根据商品ID和店铺ID更新多个商品
	 * @param array $fieldData array('字段' => '值') 需要更新的信息
	 * @param array $productIds 需要更新的商品ID列表
	 * @param integer $objectId 需要更新的商品所在店铺ID
	 * @return mixed boolean (false:操作失败) | integer 返回受影响的行数
	 */
	function updateProducts($fieldData, $productIds, $objectId) {
		$fieldData = $this->_checkProduct($fieldData);
		if (!$fieldData || !$productIds || !$objectId) return false;
		$fieldData['modifiedtime'] = $GLOBALS['modifiedtime'];
		return $this->_productsDao->updates($fieldData, $productIds, $objectId);
	}
	
	/**
	 * 通过商品名称关键字搜索商品
	 * @param string $keyword 搜索的关键字
	 * @param integer $page 开始搜索的页数
	 * @param integer $perpage 搜索的条数
	 * @return mixed boolean (false:操作失败) | array array(array('字段' => '值'),...) 返回搜索到的商品列表
	 */
	function searchProductsByTitle($keyword, $page, $perpage) {
		if ($keyword == '') return false;
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->searchProductsByTitle($keyword, $start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 按搜索商品名称关键字统计商品
	 * @param string $keyword
	 * @return mixed boolean (false:操作失败) | integer 返回统计结果
	 */
	function countProductsByTitle($keyword) {
		if ($keyword == '') return false;
		return $this->_productsDao->countProductsByTitle($keyword);
	}
	
	/**
	 * 按关键字和上下架搜索商品
	 * @param integer $objectid 店铺ID
	 * @param string $keyword 商品标题关键字
	 * @param array $shelfs  array(0) 下架   array(1) 上架   array(0,1) 全部
	 * @param integer $page 分页搜索的页码，用于检索该页的起始位置
	 * @param integer $perpage 分页搜索结果的条数
	 * @return mixed boolean (false:操作失败) | array  array(array('字段' => '值'),...) 返回搜索结果列表
	 */
	function searchByShelfAndTitle($objectid, $keyword, $shelfs, $page, $perpage) {
		$objectid = intval($objectid);
		if ($keyword == '' || $objectid < 1) return false;
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->searchByShelfAndTitle($objectid, $keyword, $shelfs, $start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 按关键字和上下架统计商品
	 * @param integer $objectid 店铺ID
	 * @param string $keyword 商品标题关键字
	 * @param array $shelfs  array(0) 下架    array(1) 上架    array(0,1) 全部
	 * @return mixed boolean (false:操作失败) | integer 返回统计的商品数
	 */
	function countByShelfAndTitle($objectid, $keyword, $shelfs) {
		$objectid = intval($objectid);
		if ($keyword == '' || $objectid < 1) return false;
		return $this->_productsDao->countByShelfAndTitle($objectid, $keyword, $shelfs);
	}
	
	/**
	 * 接上下架获取商品
	 * @param integer $objectid 店铺ID
	 * @param array $shelfs  array(0) 下架     array(1) 上架     array(0,1) 全部
	 * @param integer $page  执行分页查询的页码
	 * @param integer $perpage 执行查询的条数
	 * @return mixed boolean (false:操作失败) | array  array(array('字段' => '值'),...) 返回搜索结果列表
	 */
	function searchByShelf($objectid, $shelfs, $page, $perpage) {
		$objectid = intval($objectid);
		if ($objectid < 1) return false;
		list($start, $perpage) = $this->_filterPage($page, $perpage);
		$result = $this->_productsDao->searchByShelf($objectid, $shelfs, $start, $perpage);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 按店铺ID和上下架统计商品
	 * @param integer $objectId 对象ID 此处为店铺ID
	 * @param array $shelfs array(0) 下架    array(1) 上架    array(0,1) 全部
	 * @return mixed boolean (false:操作失败) | integer 返回统计的商品数
	 */
	function countByObjectIdAndShelfs($objectId, $shelfs) {
		$objectId = intval($objectId);
		if ($objectId < 1) return false;
		return $this->_productsDao->countByObjectIdAndShelfs($objectId, $shelfs);
	}
	
	/**
	 * 过滤查询分页参数
	 * @param integer $page   查询的页码
	 * @param integer $perpage 查询的条数
	 * @return array  array(start,//查询开始的位置<br/>
	 * 						perpage,//查询的条数<br/>
	 * 						page,//需要查询的页数<br/>
	 * 					)
	 */
	function _filterPage($page, $perpage) {
		$page = intval($page) ? intval($page) : 1;
		$start = ($page - 1) * $perpage;
		$start = intval($start);
		$perpage = intval($perpage);
		return array($start, $perpage, $page);
	}
	
	/**
	 * 管理员删除商品
	 * @attention 仅为后台提供
	 * @param array $productIds  需要删除的商品ID列表
	 * @return mixed boolean (false:操作失败) | integer 返回影响的行数
	 */
	function deletesForAdmin($productIds) {
		if (!S::isArray($productIds)) return false;
		return $this->_productsDao->deletesForAdmin($productIds);
	}
	
	
	/**
	 * 检查商品字段与值
	 * @access private
	 * @param array $fieldData array('字段' => '值') 需要过滤的商品信息
	 * @return mixed boolean (false:操作失败) | array array('字段' => '值');
	 */
	function _checkProduct($fieldData) {
		if (!S::isArray($fieldData)) return false;
		$fields = array();
		(isset($fieldData['hits'])) && $fields['hits'] = intval($fieldData['hits']);
		(isset($fieldData['objectid'])) && $fields['objectid'] = intval($fieldData['objectid']);
		(isset($fieldData['title'])) && $fields['title'] = trim($fieldData['title']);
		(isset($fieldData['description'])) && $fields['description'] = $fieldData['description'];
		(isset($fieldData['recommend'])) && $fields['recommend'] = intval($fieldData['recommend']);
		(isset($fieldData['market_price'])) && $fields['market_price'] = trim($fieldData['market_price']);
		(isset($fieldData['member_price'])) && $fields['member_price'] = trim($fieldData['member_price']);
		(isset($fieldData['shelf'])) && $fields['shelf'] = intval($fieldData['shelf']);
		(isset($fieldData['photoid'])) && $fields['photoid'] = intval($fieldData['photoid']);
		return $fields;
	}
	
	/**
	 * 绑定商品封页图片到商品
	 * @access private
	 * @param array $products array(array('字段' => '值')) 等待绑定商品图片的商品列表
	 * @return array array(array('字段' => '值')) 返回绑定商品图片后的的商品列表
	 */
	function _bindPhotos($products) {
		if (!S::isArray($products)) return array();
		$photoIds = array();
		foreach ($products as $v) {
			($v['photoid']) && $photoIds[] = $v['photoid'];
		}
		$photoService = $this->getPhotossService();
		$photos = $photoService->getPhotosByIds($photoIds);
		$tmp = array();
		foreach ($products as $v) {
			if (isset($photos[$v['photoid']])) {
				$v['photopath'] = $GLOBALS['db_attachname'] . '/' . $photos[$v['photoid']]['path'];
			} else {
				$v['photopath'] = 'mode/dianpu/images/default.jpg'; //default info
			}
			$tmp[] = $v;
		}
		return $tmp;
	}
	
	/**
	 * 根据输入的$ids的顺序排序列表
	 * @access private
	 * @param array $ids 需要排序的参照
	 * @param array $result 待排序的数据集
	 * @return array array('productid' => array('字段' => '值'),...) 返回按照$ids排序后的结果
	 */
	function _orderResult($ids, $result) {
		$_result = array();
		foreach ($ids as $key => $value) {
			$_result[$value] = array();
		}
		foreach ($result as $key => $value) {
			$_result[$value['productid']] = $value;
		}
		foreach ($_result as $key => $value) {
			if (empty($value)) unset($_result[$key]);
		}
		return $_result;
	}
	
	/**
	 * 根据店铺ID随机获得推荐商品
	 * @param array $objectIds 店铺ID
	 * @return mixed boolean (false:操作失败) | array array(array('字段' => '值'),...) 返回随机商品列表
	 */
	function getRandRecommentProduct($objectIds) {
		if (!S::isArray($objectIds)) return false;
		$result = $this->_productsDao->getsByRand($objectIds);
		return $this->_bindPhotos($result);
	}
	
	/**
	 * 获得商品图片的DAO服务对象
	 * @return PW_Dianpu_PhotosDB 
	 */
	function getPhotossService() {
		return DP::loadClass("photosservice", "service");
	}
	
	/**
	 * 获得商品的点击率的DAO服务对象
	 * @return PW_ProducthitsService 
	 */
	function getProducthitsService() {
		return DP::loadClass("producthitsservice", "service");
	}
	
	/**
	 * 获取商品DAO服务
	 * @return PW_Dianpu_ProductsDB 
	 */
	function _getProductsDao() {
		return DP::loadDB("dianpu_products");
	}
}