<?php
!defined('M_P') && exit('Forbidden');

/**
 * 店铺的服务层
 * @author 
 *
 */
class PW_DianpuService {
	
	/**
	 * 更新店铺的点击率
	 * @param integer $dianpuid 店铺ID
	 * @param integer $num  店铺点击数
	 * @return integer 返回受影响的行数
	 */
	function updateDianpuHits($dianpuid, $num = 1) {
		if (!$dianpuid || !$num) return false;
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->updateDianpuHits($dianpuid, $num);
	}
	
	/**
	 * 根据分类统计店铺数（包含未审核和关闭的店铺）
	 * @return array  array('categoryid'=>'','parentid'=>'','total'=>'')
	 */
	function statDianpuByCategory() {
		$_dianpuDB = $this->_getDianpuDB();
		$_stats = $_dianpuDB->statDianpuByCategory();
		$_tmp = array();
		foreach ($_stats as $key => $value) {
			$_tmp[$value['categoryid']] = $value;
		}
		return $_tmp;
	}
	
	/**
	 * 根据分类ID查询该分类下的店铺数（包含未审核和关闭的店铺）
	 * 
	 * 在后台删除分类的时候，需要先判断下该分类下是否已经存在有店铺，并且，该分类下的店铺状态不受限制
	 * 
	 * @param integer $categoryID 分类ID
	 * @return integer 总数
	 */
	function getDianpuCountByCategoryid($categoryID) {
		$categoryId = intval($categoryID);
		if ($categoryId < 1) return false;
		$_dianpuDB = $this->_getDianpuDB();
		$_result = $_dianpuDB->statDianpuByCategory($categoryId);
		return $_result['total'];
	}
	
	/**
	 * 移动店铺分类
	 * @param array $fieldData  array('parentid' => '', 'categoryid' => '') 需要移动的分类的信息
	 * @return integer 返回受影响的行数
	 */
	function moveDianpuCategory($fieldData) {
		if (!is_array($fieldData) || empty($fieldData)) return false;
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->updateDianpuCategory($fieldData);
	}
	
	/**
	 * 根据店铺ID获得店铺信息
	 * @param integer $id 店铺ID
	 * @return PW_DianpuModule 
	 */
	function getDianpu($id) {
		$_dianpuDB = $this->_getDianpuDB();
		$_tmp = $_dianpuDB->getDianpuById($id);
		if (!$_tmp) return false;
		return $this->_fieldDataToModule($this->_buildResult($_tmp, true));
	}
	
	/**
	 * 根据用户名获得店铺信息
	 * @param string $username 店铺的店主名字
	 * @return PW_DianpuModule 
	 */
	function getDianpuByUser($username) {
		$_dianpuDB = $this->_getDianpuDB();
		$dianpu = $_dianpuDB->getDianpuByUsername($username);
		if (!$dianpu) return false;
		return $this->_fieldDataToModule($this->_buildResult($dianpu, true));
	}
	
	/**
	 * 根据商家ID获得商家信息
	 * @param integer $shangjiaid  商家ID
	 * @return array array('字段' => '值') 返回商家信息  
	 */
	function getShangjia($shangjiaid) {
		if (!$shangjiaid) return false;
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->getShangjia($shangjiaid);
	}
	
	/**
	 * 通过ID获得店铺信息列表
	 * @param array $ids  店铺ID列表
	 * @param boolean $order 是否按照ID顺序输出
	 * @return array array(array('字段' => '值'),...) 店铺信息列表  
	 */
	function getDianpuListByIds($ids, $order = false) {
		if (!is_array($ids) || empty($ids)) return array();
		$_dianpuDB = $this->_getDianpuDB();
		$_tmp = $_dianpuDB->getDianpuListByIds($ids);
		if ($order) return $this->_orderResult($ids, $this->_buildDianpuResultList($_tmp));
		return $this->_buildDianpuResultList($_tmp);
	}
	
	/**
	 * 店铺搜索扩展支持
	 * @param array $conditions	 $conditions[$field] = array($value, 'like/eq');
	 * @param integer $start 开始搜索的位置
	 * @param integer $perpage 需要搜索的条数
	 * @return array array(array('字段' => '值'),...) 返回店铺信息列表        
	 */
	function search($conditions, $start = '0', $perpage = '0') {
		$_dianpuDB = $this->_getDianpuDB();
		$_tmp = $_dianpuDB->search($conditions, $start, $perpage);
		return $this->_buildDianpuResultList($_tmp);
	}
	
	/**
	 * 店铺搜索
	 * @param string $title  店铺标题
	 * @param string $owner  店铺店主
	 * @param integer $area  店铺所在地区ID
	 * @param integer $parentid 店铺所属行业ID
	 * @param integer $category  店铺所属行业分类ID
	 * @param integer $start  需要显示的店铺列表开始搜索位置
	 * @param integer $perpage  需要显示的店铺列表的条数
	 * @return array  array(array('字段' => '值'), ...)店铺信息列表  
	 */
	function searchForAdmin($title = '', $owner = '', $area = '', $category = '', $parentid = '', $start = '0', $perpage = '0') {
		$conditions = array();
		$conditions['ifcheck'] = array('1', 'eq');
		if ($title) $conditions['title'] = array($title, 'like');
		if ($owner) $conditions['username'] = array($owner, 'like');
		if ($area) $conditions['areaid'] = array($area, 'eq');
		if ($category) $conditions['categoryid'] = array($category, 'eq');
		if ($parentid) $conditions['parentid'] = array($parentid, 'eq');
		return $this->search($conditions, $start, $perpage);
	}
	
	/**
	 * 前台搜索服务
	 * @param integer $area  店铺所在地区ID
	 * @param integer $parentid 店铺所属行业ID
	 * @param integer $category 店铺所属行业分类ID
	 * @param array $checkway 店铺的认证方法
	 * @param string $title 搜索的店铺名字
	 * @param string $orderby 店铺显示的排序方式
	 * @param string $desc 店铺的排序顺序，in_array('asc', 'desc')
	 * @param integer $start 开始搜索的位置
	 * @param integer $perpage 搜索的条数
	 * @return array array('dianpuid' => array('字段' => '值'),...) 返回搜索的店铺列表  
	 */
	function searchForList($area = '', $parentid = '', $category = '', $checkway = array(), $title = '', $orderby = '', $desc = '', $start = '0', $perpage = '0') {
		$conditions = array();
		$conditions['state'] = array('1', 'eq');
		$conditions['ifcheck'] = array('1', 'eq');
		if ($area) $conditions['areaid'] = array($area, 'eq');
		if ($parentid) $conditions['parentid'] = array($parentid, 'eq');
		if ($category) $conditions['categoryid'] = array($category, 'eq');
		if ($title) $conditions['title'] = array($title, 'like');
		if ($checkway) {
			foreach ((array) $checkway as $key => $value) {
				$conditions['checkway'][] = array($value, 'like');
			}
		}
		if ($conditions) {
			$_dianpuDB = $this->_getDianpuDB();
			$dianpus = $_dianpuDB->searchByIndex($conditions, $orderby, $desc, $start, $perpage);
			return $this->getDianpuListByDianpuIndex($dianpus);
		}
		return array();
	}
	
	/**
	 * 前台搜索统计服务
	 * @param integer $area 搜索统计的地区信息
	 * @param integer $parentid 搜索统计的行业ID
	 * @param integer $category 搜索统计的分类ID
	 * @param string  $checkway 搜索统计的店铺认证方式
	 * @param string  $title 搜索统计的店铺名 （模糊查询）
	 * @param string  $orderby 搜索统计的排序规则
	 * @return mixed boolean  (false:操作失败) | integer 返回统计的总数
	 */
	function getCountForList($area = '', $parentid = '', $category = '', $checkway = '', $title = '', $orderby = '') {
		$conditions = array();
		$conditions['state'] = array('1', 'eq');
		$conditions['ifcheck'] = array('1', 'eq');
		if ($area) $conditions['areaid'] = array($area, 'eq');
		if ($parentid) $conditions['parentid'] = array($parentid, 'eq');
		if ($category) $conditions['categoryid'] = array($category, 'eq');
		if ($title) $conditions['title'] = array($title, 'like');
		if ($checkway) {
			foreach ((array) $checkway as $key => $value) {
				$conditions['checkway'][] = array($value, 'like');
			}
		}
		if ($conditions) {
			$_dianpuDB = $this->_getDianpuDB();
			return $_dianpuDB->searchByIndexCount($conditions);
		}
		return false;
	}
	
	/**
	 * 后台搜索末审核店铺
	 * @param string $title 店铺标题
	 * @param string $owner  店主名字
	 * @param integer $start 开始搜索的位置
	 * @param integer $perpage 搜索的类表
	 * @return array  array(array('字段' => '值'),...) 返回店铺信息列表     
	 */
	function searchForVerify($title = '', $owner = '', $start = '0', $perpage = '0') {
		$conditions = array();
		if (!empty($title)) $conditions['title'] = array($title, 'like');
		if (!empty($owner)) $conditions['username'] = array($owner, 'like');
		$conditions['ifcheck'] = array(0, 'eq');
		return $this->search($conditions, $start, $perpage);
	}
	
	/**
	 * 统计后台末审核店铺
	 * @param string  $title  店铺标题
	 * @param string $owner 店铺店主名字
	 * @return integer 返回统计总数
	 */
	function getCountByVerify($title = '', $owner = '') {
		$conditions = array();
		if (!empty($title)) $conditions['title'] = array($title, 'like');
		if (!empty($owner)) $conditions['username'] = array($owner, 'like');
		$conditions['ifcheck'] = array(0, 'eq');
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->count($conditions);
	}
	
	/**
	 * 获得未审核店铺列表
	 * @param integer $start  开始搜索的位置
	 * @param integer $perpage   搜索的条数
	 * @return array  array(array('字段' => '值'),...) 返回店铺搜索的店铺列表信息
	 */
	function getUncheckDianpuList($start = '0', $perpage = '0') {
		$conditions['ifcheck'] = array('0', 'eq');
		$_dianpuDB = $this->_getDianpuDB();
		$_tmp = $_dianpuDB->search($conditions, $start, $perpage);
		return $this->_buildDianpuResultList($_tmp);
	}
	
	/**
	 * 统计没有认证的店铺数
	 * @return integer 返回统计结果
	 */
	function countUncheckDianpuList() {
		$conditions['ifcheck'] = array('0', 'eq');
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->count($conditions);
	}
	
	/**
	 * 统计总数
	 * @param string $title  店铺标题
	 * @param string $owner  店铺店主名字
	 * @param integer $area  店铺地区ID
	 * @param integer $category  店铺行业分类ID
	 * @param integer $parentid 店铺行业ID
	 * @return integer 返回统计结果
	 */
	function getCount($title = '', $owner = '', $area = '', $category = '', $parentid = '') {
		$conditions = array();
		$conditions['ifcheck'] = array('1', 'eq');
		if (!empty($title)) $conditions['title'] = array($title, 'like');
		if (!empty($owner)) $conditions['username'] = array($owner, 'like');
		if (!empty($area)) $conditions['areaid'] = array($area, 'eq');
		if (!empty($category)) $conditions['categoryid'] = array($category, 'eq');
		if ($parentid) $conditions['parentid'] = array($parentid, 'eq');
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->count($conditions);
	}
	
	/**
	 * 随机返回一条推荐店铺数据
	 * @param string $recommendType  推荐显示 
	 * @param integer $num 显示的条数
	 * @return array  array('dianpuid' => array('字段' => '值'),...) 获得店铺信息   
	 */
	function getDianpuByRand($recommendType, $num = 10) {
		$_groupservice = $this->_getGroupService();
		$_groups = $_groupservice->getDianpushowGroups($recommendType);
		if (!$_groups) return array();
		$_dianpuDB = $this->_getDianpuDB();
		$dianpus = $_dianpuDB->searchByRandWithRecommend($num, $_groups);
		return $this->getDianpuListByDianpuIndex($dianpus);
	}
	
	/**
	 * 根据店铺id数组，获得店铺详情列表
	 * @param array $dianpus array('字段' => '值') 店铺ID信息列表  
	 * @return array  array('dianpuid' => array('字段' => '值'),...)  返回店铺信息列表
	 */
	function getDianpuListByDianpuIndex($dianpus) {
		$_tmp = array();
		foreach ($dianpus as $key => $value) {
			$_tmp[$value['dianpuid']] = $value['dianpuid'];
		}
		$_dianpuList = $this->getDianpuListByIds($_tmp);
		foreach ($_dianpuList as $key => $value) {
			if ($_tmp[$value['dianpuid']]) $_tmp[$value['dianpuid']] = $value;
		}
		return $_tmp;
	}
	
	/**
	 * 添加一个店铺
	 * @param PW_DianpuModule $dianpu  需要添加的店铺信息
	 * @return integer 返回插入返回的ID
	 */
	function insertDianpu($dianpu) {
		$fieldData = $this->_moduleToFieldData($dianpu);
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->insertDianpu($fieldData);
	}
	
	/**
	 * 更新店铺信息
	 * @param PW_DianpuModule $dianpu 需要更新的店铺信息
	 * @return boolean
	 */
	function updateDianpu($dianpu) {
		$fieldData = $this->_moduleToFieldData($dianpu);
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->updateDianpu($fieldData);
	}
	
	/**
	 * 根据店铺内容ID删除店铺内容
	 * @param integer $cid 店铺的内容ID
	 * @return mixed boolean (false:非法操作) | integer 返回受影响的行数
	 */
	function deleteDianpuContentByCid($cid) {
		if (!$cid) return false;
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->deleteContentByCid($cid);
	}
	
	/**
	 * 批量删除店铺
	 * @param array $ids 需要批量删除的店铺ID列表
	 * @return boolean
	 */
	function deleteDianpuMulit($ids) {
		if (!is_array($ids) || empty($ids)) return false;
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->deleteDianpuMulit($ids);
	}
	
	/**
	 * 根据店铺ID关闭店铺
	 * @param integer $id 需要关闭的店铺ID
	 * @return integer 返回更新影响的行数
	 */
	function closeDianpu($id) {
		$_dianpuDB = $this->_getDianpuDB();
		$_fieldData = array('dianpuid' => $id, 'state' => '0');
		return $_dianpuDB->updateDianpu($_fieldData);
	}
	
	/**
	 * 根据店铺ID开启店铺
	 * @param integer $id 需要开启的店铺ID
	 * @return integer 返回更新影响的行数
	 */
	function openDianpu($id) {
		$_dianpuDB = $this->_getDianpuDB();
		$_fieldData = array('dianpuid' => $id, 'state' => '1');
		return $_dianpuDB->updateDianpu($_fieldData);
	}
	
	/**
	 * 批量认证店铺
	 * @param array $ids 需要批量认证的店铺ID列表
	 * @param integer $ifcheck (0/1) 认证类型
	 * @return integer 返回实行店铺认证操作影响的行数
	 */
	function checkDianpuMulit($ids, $ifcheck) {
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->checkDianpuMulit($ids, $ifcheck);
	}
	
	/**
	 * 添加店铺说明内容
	 * @param array $dianpu  需要添加的店铺说明内容
	 * @return boolean
	 */
	function insertDianpuContent($dianpu) {
		if (!is_object($dianpu)) return false;
		$_dianpuDB = $this->_getDianpuDB();
		if (!$contentid = $_dianpuDB->insertContent($this->_moduleToFieldData($dianpu))) return false;
		return $this->_setDianpuContentorder($dianpu, $contentid);
	}
	
	/**
	 * 更新店铺内容
	 * @param array $dianpu  店铺说明内容
	 * @param integer $contentid 店铺说明ID
	 * @return boolean
	 */
	function updateDianpuContent($dianpu, $contentid) {
		if (!is_object($dianpu)) return false;
		$_dianpuDB = $this->_getDianpuDB();
		if (!$_dianpuDB->updateContent($this->_moduleToFieldData($dianpu), $contentid)) return false;
		return $this->_setDianpuContentorder($dianpu, $contentid);
	}
	
	/**
	 * 根据店铺ID设置店铺过期，同时设置推荐类型为不推荐
	 * @param array $dianpuids
	 * @return boolean
	 */
	function setOverdueDianpu($dianpuids) {
		if (!is_array($dianpuids)) return false;
		$_dianpuDB = $this->_getDianpuDB();
		if (!$_dianpuDB->updateDianpuOvertime($dianpuids)) return false;
		return true;
	}
	
	/**
	 * 根据店铺说明内容ID获得店铺说明内容
	 * @param integer $contentid
	 * @return mixed boolean (false:操作失败) | array array('字段' => '值')
	 */
	function getDianpuContent($contentid) {
		if (!$contentid) return false;
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->getDianpuContent($contentid);
	}
	
	/**
	 * 删除店铺说明内容
	 * @param array $contentids  店铺说明ID
	 * @return mixed boolean (false:操作失败) | integer 返回受影响的行数
	 */
	function deleteDianpuContents($contentids) {
		if (!$contentids) return false;
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->deleteContents($contentids);
	}

	/**
	 * 根据店铺ID批量关闭店铺
	 * @param array $ids  店铺ID列表
	 * @param string $state  状态信息
	 * @return integer
	 */
	function closeDianpuMulit($ids, $state) {
		$_dianpuDB = $this->_getDianpuDB();
		return $_dianpuDB->stateDianpuMulit($ids, $state);
	}
	
	/**
	 * 给结果集按照IDS排序输出
	 * @access private
	 * @param array $ids   排序列表
	 * @param array $result array(array('字段' => '值')...) 需要排序的结果集     
	 * @return array array('dianpuid' => array('字段' => '值'),...)  根据ids排序后的结果集返回
	 */
	function _orderResult($ids, $result) {
		$_result = array();
		foreach ($ids as $key => $value) {
			$_result[$value] = array();
		}
		foreach ($result as $key => $value) {
			$_result[$value['dianpuid']] = $value;
		}
		return $_result;
	}
	
	/**
	 * 添加店铺内容排序
	 * @access private
	 * @param PW_DianpuModule $dianpu 需要店铺店铺内容的店铺信息
	 * @param integer $contentid  店铺说明ID
	 * @return boolean 
	 */
	function _setDianpuContentorder($dianpu, $contentid) {
		if (!$contentid) return false;
		$_dianpuDB = $this->_getDianpuDB();
		$_contentorder = $dianpu->contentorder;
		$_contentorder[$contentid]['t'] = $dianpu->titles;
		$dianpu->titlesorder && $_contentorder[$contentid]['o'] = $dianpu->titlesorder;
		$dianpu->setContentorder($_contentorder);
		return $_dianpuDB->updateDianpu($this->_moduleToFieldData($dianpu));
	}
	
	/**
	 * 获得扩展的字段封装
	 * @access private
	 * @param PW_DianpuModule $dianpu
	 * @return array array('phone' => '', 'overtime' => '');
	 */
	function _getExtendField($dianpu) {
		$_result = array();
		$_result['phone'] = (!is_null($dianpu->areanumber) && !is_null($dianpu->phone)) ? $dianpu->areanumber . '-' . $dianpu->phone : null;
		$_result['overtime'] = $dianpu->overtime ? PwStrtoTime($dianpu->overtime) : 0;
		return $_result;
	}
	
	/**
	 * 获得扩展的module封装
	 * @access private
	 * @param array $fieldData  输入需要过滤的数组信息
	 * @return array array('字段' => '值')
	 */
	function _getExtendModule($fieldData) {
		$_result = array();
		$_result['phone'] = substr($fieldData['phone'], (strpos($fieldData['phone'], '-') + 1));
		$_result['areanumber'] = substr($fieldData['phone'], '0', strpos($fieldData['phone'], '-'));
		$_result['checkwayhtml'] = $fieldData['checkwayhtml'];
		$_result['contentorder'] = $fieldData['contentorder'];
		$_result['category'] = $fieldData['category'];
		$_result['parent'] = $fieldData['parent'];
		$_result['group'] = $fieldData['group'];
		return $_result;
	}
	
	/**
	 * 将PW_DianpuModule转化成数组输出
	 * @access private
	 * @param PW_DianpuModule $dianpu
	 * @return array array('字段' => '值', ...)
	 */
	function _moduleToFieldData($dianpu) {
		$_extends = $this->_getExtendField($dianpu);
		$struct = $this->getTableStruct();
		$_result = array();
		foreach ($struct as $key => $value) {
			$_result[$value] = $dianpu->{$value};
		}
		foreach ($_extends as $key => $value) {
			$_result[$key] = $value;
		}
		return $_result;
	}
	
	/**
	 * 将数组数据转化成PW_DianpuModule输出
	 * @access private 
	 * @param array $fieldData
	 * @return PW_DianpuModule 
	 */
	function _fieldDataToModule($fieldData) {
		$dianpu = DP::loadClass('dianpumodule', 'module'); /* @var $dianpu PW_DianpuModule */
		$_extends = $this->_getExtendModule($fieldData);
		$struct = $this->getTableStruct();
		foreach ($struct as $key => $value) {
			if (array_key_exists($value, get_class_vars('PW_DianpuModule'))) $dianpu->{$value} = $fieldData[$value];
		}
		foreach ($_extends as $key => $value) {
			if (array_key_exists($key, get_class_vars('PW_DianpuModule'))) $dianpu->{$key} = $value;
		}
		return $dianpu;
	}
		
	/**
	 * 构造返回的结果集
	 * @access private
	 * @param array $result  array(array('字段' => '值'),...) 需要构造的数组
	 * @return array array(array('字段' => '值'),...) 返回构造的结果
	 */
	function _buildDianpuResultList($result) {
		$_tmp = array();
		foreach ((array) $result as $key => $value) {
			$_tmp[$key] = $this->_buildResult($value, true);
		}
		return $_tmp;
	}
	
	/**
	 * 组装结果集
	 * @access private
	 * @param array $fieldData  组装的结果集
	 * @param boolean $buildField  是否组装行业信息和地区信息
	 * @return array array('字段' => '值');
	 */
	function _buildResult($fieldData, $buildField = false) {
		if ($fieldData['createtime']) $fieldData['createtime'] = get_date($fieldData['createtime'], 'Y-m-d H:i');
		if ($fieldData['overtime']) $fieldData['overtime'] = get_date($fieldData['overtime'], 'Y-m-d H:i');
		if (isset($fieldData['groupid']) && $fieldData['groupid']) {
			$groups = L::config('dianpuGroups', 'dianpu_groups');
			$_groupsInfo = $groups[$fieldData['groupid']];
			$fieldData['group'] = $_groupsInfo;
		}
		if (isset($fieldData['logo']) && $fieldData['logo'] && $_groupsInfo['configinfo']['uploadlogo']) {
			$_tmp = geturl($fieldData['logo'], '', true);
			$fieldData['logo'] = $_tmp[0];
		} else {
			$fieldData['logo'] = 'mode/dianpu/images/default.jpg';
		}
		if (isset($fieldData['checkway']) && $fieldData['checkway']) {
			$fieldData['checkwayhtml'] = getDianpuCheckWay($fieldData['checkway'], 'img');
		}
		if (isset($fieldData['categoryid']) && $fieldData['categoryid'] && $buildField) {
			$category = DP::loadClass('categoriesservice', 'service');
			/* @var $category PW_CategoriesService */
			$fieldData['category'] = $category->getCategory($fieldData['categoryid']);
			$fieldData['parent'] = $category->getCategory($fieldData['category']['parentid']);
			$fieldData['category'] = $fieldData['category']['name'];
			$fieldData['parent'] = $fieldData['parent']['name'];
		}
		if (isset($fieldData['areaid']) && $fieldData['areaid'] && $buildField) {
			$area = DP::loadClass('areasservice', 'service');
			/* @var $area PW_AreasService */
			$fieldData['area'] = $area->getArea($fieldData['areaid']);
			$fieldData['area'] = $fieldData['area']['name'];
		}
		return $fieldData;
	}
	
	/**
	 * 获得店铺表结构
	 * @return array 
	 */
	function getTableStruct() {
		$dianpuDB = $this->_getDianpuDB();
		return $dianpuDB->getStruct();
	}
	
	/**
	 * 获得店铺组service
	 * @access private
	 * @return PW_GroupService 
	 */
	function _getGroupService() {
		return DP::loadClass('groupservice', 'service');
	}
	
	/**
	 * 获得店铺组DAO
	 * @access private
	 * @return PW_Dianpu_DianpuDB 
	 */
	function _getDianpuDB() {
		return DP::loadDB('dianpu_dianpu');
	}
}
?>