<?php
!defined('P_W') && exit('Forbidden');
require_once MODE . '/base/basehelper.php';

/**
 * 店铺行业分类Helper层实现
 * @author L.iuHu.I
 */
class PW_CategoriesHelper extends BaseHelper {
	
	/**
	 * 新增行业和分类
	 * @param array $oldCategories 需要更新的大/小分类
	 * @param array $newParents 新增的大分类
	 * @param array $newChildren 新增的小分类
	 * @return boolean 返回操作结果
	 */
	function addCategories($oldCategories, $newParents, $newChildren) {
		$categoriesService = $this->getCategoriesService();
		list($sorts, $names) = array($newParents['sort'], $newParents['name']);
		if (!S::isArray($sorts)) $sorts = array();
		if (!S::isArray($names)) $names = array();
		$tmp = array();
		foreach ($names as $key => $name) {
			if ($name) $tmp[] = array('name' => $name, 'sort' => intval($sorts[$key]));
		}
		$categoriesService->addCategories($tmp);
		
		if (!S::isArray($newChildren)) $newChildren = array();
		$fields = array();
		foreach ($newChildren as $parentId => $childs) {
			list($sorts, $names) = array($childs['sort'], $childs['name']);
			foreach ($names as $key => $name) {
				if ($name) $fields[] = array('parentid' => $parentId, 'name' => $name, 
					'sort' => intval($sorts[$key]));
			}
		}
		$categoriesService->addCategories($fields);
		
		foreach ($oldCategories as $categoryId => $category) {
			$categoryId = intval($categoryId);
			if ($categoryId < 1) continue;
			if ($category['name']) $categoriesService->updateCategory(array('sort' => intval($category['sort']), 
				'name' => $category['name']), $categoryId);
		}
		Error::addError('操作成功');
		return true;
	}
	
	/**
	 * 根椐行业获取分类
	 * @param integer $parentId 行业ID
	 * @return boolean (false:获取失败) | array 返回行业数据
	 */
	function getCategoriesByParentId($parentId) {
		list($categories) = $this->getCategories();
		if (!$categories) return false;
		return isset($categories[$parentId]) ? $categories[$parentId] : false;
	}
	
	/**
	 * 获取全部行业与分类
	 * @return array
	 */
	function getCategories() {
		$categoriesService = $this->getCategoriesService();
		$categories = $categoriesService->getCategories();
		return $this->_buildCategories($categories);
	}
	
	/**
	 * 获取全部行业与分类无缓存
	 * @return array 
	 */
	function getCategoriesNoCache() {
		$categoriesService = $this->getCategoriesService();
		$categories = $categoriesService->getCategoriesNoCache();
		return $this->_buildCategories($categories);
	}
	
	/**
	 * 组装全部行业与分类
	 * @param array $categories
	 * @return boolean (false:操作失败) | array 
	 */
	function _buildCategories($categories) {
		if (!$categories) return false;
		$parent = array();
		foreach ($categories as $key => $category) {
			if ($category['parentid'] <= 0) continue;
			$parent[$category['parentid']][] = $category;
			unset($categories[$key]);
		}
		$tmp = array();
		foreach ($categories as $category) {
			if ($parent[$category['categoryid']]) $category['children'] = $parent[$category['categoryid']];
			$tmp[$category['categoryid']] = $category;
		}
		return array($tmp, $categories);
	}
	
	/**
	 * 获取某个分类信息
	 * @param integer $categoryId 分类ID
	 * @return mixed integer (0:操作失败) | array 返回分类信息
	 */
	function getCategory($categoryId) {
		$categoryId = intval($categoryId);
		if ($categoryId < 1) return 0;
		$categoriesService = $this->getCategoriesService();
		return $categoriesService->getCategory($categoryId);
	
	}
	
	/**
	 * 获取行业
	 * @return mixed boolean (false:操作失败) | array 返回行业信息
	 */
	function getParentCategories() {
		list($allCategories, $parentCategories) = $this->getCategories();
		if (!$parentCategories) return false;
		return $parentCategories;
	}
	
	/**
	 * 根椐行业获取默认分类
	 * @return mixed integer (0:操作失败) | array 返回分类信息
	 */
	function getDefaultParentCategory() {
		$categories = $this->getParentCategories();
		if (!$categories) return 0;
		$tmp = array();
		foreach ($categories as $c) {
			$tmp[] = $c['categoryid'];
		}
		return $tmp[0];
	}
	
	/**
	 * 获取行业下拉选项
	 * @param string $select 设置默认选中的项
	 * @return string 返回行业下拉
	 */
	function getParentOption($select) {
		$parentCategories = $this->getParentCategories();
		if (!$parentCategories) return '';
		$tmp = array();
		foreach ($parentCategories as $c) {
			$tmp[$c['categoryid']] = $c['name'];
		}
		return $this->_buildOption($tmp, $select);
	}
	
	/**
	 * 获取分类下拉选项
	 * @param integer $parentId 行业ID
	 * @param string $select 默认选中的分类
	 * @return string 返回分类下拉
	 */
	function getChildOption($parentId, $select) {
		$childCategories = $this->getCategoriesByParentId($parentId);
		if (!S::isArray($childCategories['children'])) return '';
		$tmp = array();
		foreach ($childCategories['children'] as $c) {
			$tmp[$c['categoryid']] = $c['name'];
		}
		return $this->_buildOption($tmp, $select);
	}
	
	/**
	 * 移动分类
	 * @param $source
	 * @param $target
	 * @param $sourcechild
	 * @param $targetchild
	 * @return array array(boolean, string) 返回移动结果
	 */
	function moveCategory($source, $target, $sourcechild, $targetchild) {
		if ($source == $target) return array(false, '请选择不同的行业');
		$source = intval($source);
		$target = intval($target);
		$sourcechild = intval($sourcechild);
		$targetchild = intval($targetchild);
		if ($source < 1 || $target < 1 || $sourcechild < 1 || $targetchild < 1) return array(false, '数据有误');
		$categoriesService = $this->getCategoriesService();
		if (!$categoriesService->moveCategory(array('parentid' => $target), $sourcechild)) return array(false, '操作失败');
		$dianpuService = $this->getDianpuService();
		$dianpuService->moveDianpuCategory(array('parentid' => $target, 'categoryid' => $sourcechild));
		return array(true, '操作成功');
	}
	
	/**
	 * 删除行业或分类
	 * @param integer $categoryId 行业或分类ID
	 * @return array  array(boolean , string) 返回操作结果
	 */
	function deleteCategory($categoryId) {
		$categoryId = intval($categoryId);
		if ($categoryId < 1) return array(false, '数据有误');
		$categoriesService = $this->getCategoriesService();
		$_tmp = $categoriesService->getCategoriesByParentId($categoryId);
		if ($_tmp) return array(false, '必须先删除分类才能删除行业');
		$dianpuService = $this->getDianpuService();
		$_tmp = $dianpuService->getDianpuCountByCategoryid($categoryId);
		if ($_tmp) return array(false, '该分类下含有店铺信息，不能删除！');
		$_tmp = $categoriesService->deleteCategory($categoryId);
		if (!($_tmp)) return array(false, '删除失败');
		return array(true, '删除成功');
	}
	
	/*
	* 组装下拉框
	* @param array $arrays 需要构造下拉框的数组信息
	* @param string $select 设置为默认选中
	* @param boolean $isEmpty 是否使用缺省值
	* @return string 返回下拉
	*/
	function _buildOption($arrays, $select = '', $isEmpty = false) {
		if (!is_array($arrays)) return '';
		$html = '';
		($isEmpty == true) && $html .= '<option value="">请选择行业</option>';
		foreach ($arrays as $k => $v) {
			$selected = ($select == $k && $select != null) ? 'selected="selected"' : "";
			$html .= '<option value="' . $k . '" ' . $selected . '>' . $v . '</option>';
		}
		return $html;
	}
	
	/**
	 * 获取行业与分类
	 * @return array 
	 */
	function getParentAndChild() {
		$categoriesService = $this->getCategoriesService();
		return $categoriesService->getParentAndChild();
	}
	
	/**
	 * 根据分类信息统计店铺数
	 * @return array
	 */
	function statDianpyCountByCategory() {
		$dianpuService = $this->getDianpuService();
		return $dianpuService->statDianpuByCategory();
	}

}