<?php
!defined('M_P') && exit('Forbidden');
require_once MODE . '/base/basehelper.php';

/**
 * 店铺组相关Helper层实现
 * @author qiong
 */
class PW_GroupsHelper extends BaseHelper {
	
	/**
	 * 获得店铺组分页列表
	 * @param integer $page 分页获取页数
	 * @param integer $perpage 分页每页显示的条数
	 * @return array array(list, pages, start) 返回分页信息（列表，分页串，开始位置)
	 */
	function getGroupsList($page, $perpage) {
		$service = $this->getGroupsService();
		$count = $service->getDianpuGroupsCount();
		list($page, $start) = $this->_validationPage($page, $count, $perpage);
		$list = $service->getDianpuGroupsByPages($start, $perpage);
		$pages = numofpage($count, $page, ceil($count / $perpage), $this->_getGlobal('basename') . '&action=groups&');
		return array($list, $pages, $start);
	}
	
	/**
	 * 添加编辑店铺组
	 * @param string $title 店铺组名称
	 * @param array $config 店铺组配置信息
	 * @param array $style 店铺组风格
	 * @param int $groupid 店铺组ID
	 * @return boolean 返回店铺组添加是否成功
	 */
	function insertGroup($title, $config, $style, $defaultstyle, $groupid = 0) {
		$service = $this->getGroupsService();
		if (empty($title)) return $this->_addError('店铺组名称不能为空');
		if (!is_array($config) || empty($config)) return $this->_addError('店铺组配置信息不能为空');
		if (!is_array($style) || empty($style)) return $this->_addError('至少选择一个店铺组风格');
		$_styleData = array();
		foreach ($style as $key => $value) {
			if ($value == $defaultstyle) {
				$_styleData['default'] = $value;
			} else {
				$_styleData[] = $value;
			}
		}
		$config['maxproduct'] = str_replace(array('-'), '', (int) $config['maxproduct']);
		$config['maxexplain'] = str_replace(array('-'), '', (int) $config['maxexplain']);
		$_fieldData = array('title' => $title, 'configinfo' => $config, 'styles' => $_styleData, 'state' => '1');
		if (!$groupid) {
			$_fieldData['createtime'] = $this->_getGlobal('timestamp');
			$service->insertDianpuGroup($_fieldData);
			$service->updateDianpuGroupCache();
			return true;
		} else {
			$service->updateDianpuGroup($_fieldData, $groupid);
			$service->updateDianpuGroupCache();
			return true;
		}
		return false;
	}
	
	/**
	 * 根据店铺ID获得店铺组信息
	 * @param integer $gid 店铺组ID
	 * @return mixed  boolean(false:获取失败) | array array('字段' => '值') 返回店铺组信息
	 */
	function getGroupInfo($gid) {
		if (!$gid) return $this->_addError('请选择一个用户组进行编辑！');
		$service = $this->getGroupsService();
		$group = $service->getGroupInfo($gid);
		if ($group) return $group;
		return $this->_addError('获得用户组信息失败！');
	}
	
	/**
	 * 根据店铺组ID删除店铺组信息
	 * @param integer $gid 需要删除的店铺组ID
	 * @return boolean 返回删除结果
	 */
	function delete($gid) {
		if (!$gid) return $this->_addError('请选择要删除的店铺组！');  
		$dianpuService = $this->getDianpuService();
		if ($dianpuService->search(array('groupid' => array($gid, 'eq')))) return $this->_addError('该店铺组有店铺数据，您不能删除该店铺组！');
		$service = $this->getGroupsService();
		if ($service->deleteDianpuGroup($gid)) return true;
		return $this->_addError('删除店铺组操作失败');  
	}
	
	/**
	 * 获得所有可用的风格列表
	 * @return array 返回可用的风格列表
	 */
	function getStylesList() {
		$service = $this->getStylesService();
		return $service->getStyles();
	}
	
	/**
	 * 过滤输入的分页参数：页数，总数，每页的条数
	 * @access private
	 * @param integer $page 输出的页数
	 * @param integer $count 总条数
	 * @param integer $perpage 每页显示的条数
	 * @return array array(页数，开始搜索的位置)
	 */
	function _validationPage($page, $count, $perpage) {
		$page = $page > ceil($count / $perpage) ? ceil($count / $perpage) : $page;
		$page = (int) $page <= 0 ? 1 : $page;
		$start = ($page - 1) * $perpage;
		return array($page, $start);
	}
	
	/**
	 * 获得店铺风格及默认风格
	 * @access private
	 * @param array $allStyles 所有可用的店铺风格
	 * @param array $groupstyles 店铺组已有的风格
	 * @return array array(所有可用的店铺风格，默认的店铺风格）
	 */
	function _getStyles($allStyles, $groupstyles = array()) {
		if (!$allStyles) return array(array(), '');
		foreach ($allStyles as $key => $value) {
			unset($allStyles[$key]);
			$allStyles[$value['styleid']] = $value;
		}
		$_styles = array();
		//获得默认店铺风格设置
		$_default = '';
		if (is_array($groupstyles)) $_default = $groupstyles['default'];
		if (!$_default || !array_key_exists($_default, $allStyles)) {
			$_any = array_slice($allStyles, 0, 1);
			$_default = $_any[0]['styleid'];
		}
		if ($_default) $allStyles[$_default]['checked_default'] = '1';
		return array($allStyles, $_default);
	}
	
	/**
	 * 获得当前使用的风格
	 * @param array $allStyles 所有可用的店铺风格
	 * @param array $groupstyles 店铺组已有的风格
	 * @param array $current 当前用的风格
	 * @return array 获得当前应用风格
	 */
	function getCurrentStyle($allStyles, $groupstyles = array(), $current = '') {
		if (!$allStyles) return '';
		list($allStyles, $_default) = $this->_getStyles($allStyles, $groupstyles);
		if (!$current || !array_key_exists($current, $allStyles)) $current = $_default;
		return $allStyles[$current];
	}
	
	/**
	 * 获得所有可用的风格
	 * @param array $allStyles
	 * @param array $groupstyles
	 * @return array  获得所有风格信息，并标注已被选择的风格信息
	 */
	function getAllStyles($allStyles, $groupstyles = array()) {
		list($allStyles, $_default) = $this->_getStyles($allStyles, $groupstyles);
		$flag = false;
		foreach ($allStyles as $key => $value) {
			if (in_array($key, $groupstyles)) {
				$allStyles[$key]['checked'] = '1';
				$flag = true;
			}
		}
		if (!$flag && $_default) $allStyles[$_default]['checked'] = '1';
		return $allStyles;
	}
	
	/**
	 * 获得店铺组的风格
	 * @param array $allStyles
	 * @param array $groupstyles
	 * @param string $current
	 * @return array 
	 */
	function getGroupStyle($allStyles, $groupstyles = array(), $current = '') {
		list($allStyles, $_default) = $this->_getStyles($allStyles, $groupstyles);
		if (!$current || !array_key_exists($current, $allStyles)) $current = $_default;
		$_result = array();
		foreach ($groupstyles as $key => $value) {
			if (array_key_exists($value, $allStyles)) $_result[$value] = $allStyles[$value];
		}
		if (!$_result && $_default) $_result = array($_default => $allStyles[$_default]);
		if (array_key_exists($current, $_result)) $_result[$current]['checked'] = '1';
		return $_result;
	}
	
	/**
	 * 获得默认的店铺组配置信息
	 * @return array 店铺组默认的配置信息
	 */
	function getDefaultGroupConfig() {
		return array('dianpushow' => '1', 'uploadlogo' => '1', 'comments' => '1', 'productshow' => '1', 
			'wangwang' => '1', 'qq' => '1', 'maxproduct' => '12', 'maxexplain' => '5', 'recommendshow' => '1', 
			'styleid' => '1');
	}
	
	/**
	 * 获得店铺组配置信息对象
	 * @param array $configinfo 店铺组的配置信息
	 * @return stdClass  获得店铺组配置信息的对象
	 */
	function getDianpuGroupsConfig($configinfo) {
		$_config = new stdClass();
		ifcheck($configinfo['dianpushow'], 'dianpushow');
		ifcheck($configinfo['comments'], 'comments');
		ifcheck($configinfo['wangwang'], 'wangwang');
		ifcheck($configinfo['qq'], 'qq');
		ifcheck($configinfo['uploadlogo'], 'uploadlogo');
		ifcheck($configinfo['productshow'], 'productshow');
		ifcheck($configinfo['recommendshow'], 'recommendshow');
		$_config->uploadlogo_Y = $this->_getGlobal('uploadlogo_Y');
		$_config->uploadlogo_N = $this->_getGlobal('uploadlogo_N');
		$_config->comments_Y = $this->_getGlobal('comments_Y');
		$_config->comments_N = $this->_getGlobal('comments_N');
		$_config->wangwang_Y = $this->_getGlobal('wangwang_Y');
		$_config->wangwang_N = $this->_getGlobal('wangwang_N');
		$_config->qq_Y = $this->_getGlobal('qq_Y');
		$_config->qq_N = $this->_getGlobal('qq_N');
		$_config->dianpushow_Y = $this->_getGlobal('dianpushow_Y');
		$_config->dianpushow_N = $this->_getGlobal('dianpushow_N');
		$_config->productshow_Y = $this->_getGlobal('productshow_Y');
		$_config->productshow_N = $this->_getGlobal('productshow_N');
		$_config->recommendshow_Y = $this->_getGlobal('recommendshow_Y');
		$_config->recommendshow_N = $this->_getGlobal('recommendshow_N');
		return $_config;
	}
}
?>