<?php
!defined('M_P') && exit('Forbidden');
/**
 * 商家导航-用户评论内容表
 * @author xiaoxia.xuxx@2010-09-20
 * 
 */
class PW_Dianpu_CommentContentsDB extends BaseDB {
	var $_tableName = "pw_dianpu_commentcontents";
	var $_commentReplysTable = 'pw_dianpu_commentreplys';
	var $_commentsTable = 'pw_dianpu_comments';
	var $_memebrsTable = 'pw_members';
	var $_primaryKey = 'commentid';
	
	/**
	 * 插入评论
	 * @param array $fieldData 待插入的评论信息
	 * @return integer 返回插入的ID
	 */
	function insert($fieldData) {
		$insertData = $this->_clearFieldData($fieldData);
		return $this->_insert($insertData);
	}
	
	/**
	 * 更新评论
	 * @param array $fieldData 评论的需要更新的内容
	 * @param integer $commentid 评论的ID
	 * @return integer 返回更新影响的行数
	 */
	function update($fieldData, $commentid) {
		$fieldData = $this->_clearFieldData($fieldData);
		return $this->_update($fieldData, $commentid);
	}
	
	/**
	 * 更新评论的状态
	 * @param integer $commentid 需要更改状态的评论ID
	 * @param integer $states 需要更改的状态（0为未回复,1为已回复）
	 * @return integer 返回影响的行数
	 */
	function changeState($commentid, $states = 1) {
		if (!$commentid) return false;
		$commentid = intval($commentid);
		$this->_db->update("UPDATE " . $this->_tableName . ' SET state = ' . $this->_addSlashes($states) . ' WHERE commentid = ' . $this->_addSlashes($commentid));
		return $this->_db->affected_rows();
	}
	
	/**
	 * 站长后台删除一条评论
	 * @param array $commentid  评论的ID列表
	 * @return integer 返回删除影响到的记录数
	 */
	function delForAdmin($commentid) {
		if (!$commentid) return false;
		return $this->_delete($commentid);
	}
	
	/**
	 * 站长后台删除多条评论
	 * @param array $commentIDs 评论ID列表
	 * @return integer 返回删除影响到的记录数
	 */
	function delsForAdmin($commentIDs) {
		if (!S::isArray($commentIDs)) return false;
		$this->_db->update("DELETE FROM " . $this->_tableName . " WHERE commentid IN (" . $this->_getImplodeString($commentIDs) . ") ");
		return $this->_db->affected_rows();
	}
	
	/**
	 * 前台执行删除一条评论，需要判断店铺（或对象）ID
	 * @param array $commentid  评论的ID列表
	 * @param integer $objectid  该条评论对应的店铺（或者其他对象）的ID
	 * @return integer 返回删除影响到的记录数
	 */
	function delete($commentid, $objectid) {
		if (!$commentid || !$objectid) return false;
		$this->_db->update("DELETE FROM " . $this->_tableName . " WHERE commentid = " . $this->_addSlashes($commentid) . " AND objectid= " . $this->_addSlashes($objectid));
		return $this->_db->affected_rows();
	}
	
	/**
	 * 前台删除多条评论，需要判断店铺（或对象）ID
	 * @param array $commentIDs 评论ID列表
	 * @param integer $objectid 被删除的评论所对应的店铺（或者对象）的ID
	 * @return integer 返回删除影响到的记录数
	 */
	function deletes($commentIDs, $objectid) {
		if (!S::isArray($commentIDs) || !$objectid) return false;
		$this->_db->update("DELETE FROM " . $this->_tableName . " WHERE commentid IN (" . $this->_getImplodeString($commentIDs) . ") AND objectid= " . $this->_addSlashes($objectid));
		return $this->_db->affected_rows();
	}
	
	/**
	 * 删除一个商店或是删除一个商品的时候，需要同时删除该商店或是商品下的所有评论
	 * @param integer $objectid 商店ID
	 * @param integer $typeid 商品ID
	 * @return integer 返回受影响的行数
	 */
	function deleteByObject($objectid, $typeid = 0) {
		if (!$objectid) return false;
		$_where = '';
		if (intval($typeid) != 0) $_where .= ' AND typeid = ' . $this->_addSlashes($typeid);
		$this->_db->update("DELETE commentid FROM " . $this->_tableName . ' WHERE objectid = ' . $this->_addSlashes($objectid) . $_where);
		return $this->_db->affected_rows();
	}
	
	/**
	 * 根据条件搜索评论(后台搜索)
	 * @param array $params中含有的参数：
	 * array('content' => '评论内容',  
	 * 'username' => '评论发表者',  
	 * 'startdate' => '查询评论的开始时间 ', 
	 * 'enddate' => '查询评论的结束时间' ,
	 * 'isanswer' = > '是否是回复，如果是1则是回复',
	 * 'state' => '该条评论是否已经被回复，1为已经被回复',
	 * 'objectid' => '筛选评论列表的店铺（或是对象）的ID',
	 * 'typeid' => '筛选评论列表的商品ID',
	 * )
	 * @param integer $start 查询翻页的开始位置
	 * @param integer $perpage 查询每页的显示条数
	 * @return array array(array(),array()...) 返回查询到的结果
	 */
	function search($params, $start = 0, $perpage = 20) {
		$_where = $this->_filterCondition($params);
		$order = ' ORDER BY postdate DESC , commentid';
		$limit = $this->_Limit($start, $perpage);
		$query = $this->_db->query("SELECT * FROM " . $this->_tableName . " WHERE 1 " . $_where . $order . $limit);
		return $this->_getAllResultFromQuery($query);
	}
	
	/**
	 * 计算得到评论的总条数
	 * @param array $params 对应需要查询的筛选条件
	 * array('content' => '评论内容',  
	 * 'username' => '评论发表者',  
	 * 'startdate' => '查询评论的开始时间 ', 
	 * 'enddate' => '查询评论的结束时间' ,
	 * 'isanswer' = > '是否是回复，如果是1则是回复',
	 * 'state' => '该条评论是否已经被回复，1为已经被回复',
	 * 'objectid' => '筛选评论列表的店铺（或是对象）的ID',
	 * 'typeid' => '筛选评论列表的商品ID',
	 * )
	 * @return integer 返回评论的条数
	 */
	function count($params) {
		$_where = $this->_filterCondition($params);
		$rt = $this->_db->get_one("SELECT COUNT(*) AS count FROM " . $this->_tableName . " WHERE 1 " . $_where);
		return $rt['count'];
	}
	
	/**
	 * 对输入的筛选条件进行过滤并组装where查询
	 * @param array $params 筛选条件的数组
	 * array('content' => '评论内容',  
	 * 'username' => '评论发表者',  
	 * 'startdate' => '查询评论的开始时间 ', 
	 * 'enddate' => '查询评论的结束时间' ,
	 * 'isanswer' = > '是否是回复，如果是1则是回复',
	 * 'state' => '该条评论是否已经被回复，1为已经被回复',
	 * 'objectid' => '筛选评论列表的店铺（或是对象）的ID',
	 * 'typeid' => '筛选评论列表的商品ID',
	 * )
	 * @return  string 组装输出的sql查询条件
	 */
	function _filterCondition($params) {
		$_where = '';
		if (!is_array($params)) $params = array($params);
		if (isset($params['isanswer']) && is_numeric($params['isanswer'])) $_where = ' AND isanswer = ' . $this->_addSlashes(intval($params['isanswer']));
		if (!empty($params['content'])) $_where .= " AND content LIKE " . $this->_addSlashes('%' . $params['content'] . '%');
		if (!empty($params['username'])) $_where .= " AND username LIKE " . $this->_addSlashes('%' . $params['username'] . '%');
		if (!empty($params['startdate'])) {
			$startdate = strtotime(trim($params['startdate'] . ' 00:00:00'));
			$_where .= ' AND postdate >= ' . $this->_addSlashes($startdate);
		}
		if (!empty($params['enddate'])) {
			$enddate = strtotime(trim($params['enddate']) . ' 23:59:59');
			$_where .= ' AND postdate <= ' . $this->_addSlashes($enddate);
		}
		if (in_array($params['state'], array('y', 'n'))) {
			$_temp = array('y' => '1', 'n' => '0');
			$_where .= ' AND state = ' . $this->_addSlashes($_temp[$params['state']]);
		}
		if ($params['objectid']) $_where .= ' AND objectid = ' . $this->_addSlashes(intval($params['objectid']));
		if ($params['typeid']) $_where .= ' AND typeid = ' . $this->_addSlashes(intval($params['typeid']));
		return $_where;
	}
	
	/**
	 * 根据店铺ID或是商品ID获得该类的所有根级commentID
	 * @param integer $objectid 店铺ID（或者是对象ID）需要获得评论的来源
	 * @param integer $typeid 店铺中商品的ID  ----商品内容页上的评论列表的实现
	 * @param integer $start  开始输出的记录位置，用于翻页
	 * @param integer $perpage 需要输出的记录的条数，用于翻页
	 * @return array array(array(),array()...)  输出符合条件的评论列表
	 */
	function getRootidByObject($objectid, $typeid = 0, $start = 0, $perpage = 0) {
		if (!$objectid) return array();
		$_where = '';
		if (intval($typeid) != 0) $_where .= ' AND typeid = ' . $this->_addSlashes(intval($typeid));
		$_limit = $this->_Limit($start, $perpage);
		$_order = " ORDER BY postdate DESC";
		$_result = $this->_db->query("SELECT commentid FROM " . $this->_commentsTable . ' WHERE objectid = ' . $this->_addSlashes($objectid) . $_where . $_order . $_limit);
		return $this->_getAllResultFromQuery($_result);
	}
	
	/**
	 * 根据店铺ID或是商品ID获得该类的所有根级评论的条数
	 * @param integer $objectid 店铺ID（或者是对象ID）需要获得评论的来源
	 * @param integer $typeid 店铺中商品的ID  ----商品内容页上的评论列表的实现
	 * @return integer 返回符合条件的评论条数
	 */
	function getFrontCount($objectid, $typeid) {
		if (!$objectid) return array();
		$_where = '';
		if (intval($typeid) != 0) $_where .= ' AND typeid = ' . $this->_addSlashes(intval($typeid));
		$rt = $this->_db->get_one("SELECT COUNT(*) AS count FROM " . $this->_commentsTable . ' WHERE objectid = ' . $this->_addSlashes($objectid) . $_where);
		return $rt['count'];
	}
	
	/**
	 * 根据评论ID列表，获得评论的内容信息列表（根据时间倒序来） 根级评论的详细信息，
	 * @param array $ids 评论ID列表
	 * @return array array('commentid1' => array(), 'commentid2' => array()....) 返回该评论ID类表的详细信息
	 */
	function getFrontCommentsInfoList($ids) {
		if (empty($ids)) return array();
		$sql = "SELECT cn.*, re.parentid as parentid, re.rootid as rootid, m.icon as icon " . "FROM " . $this->_tableName . ' cn ' . 'LEFT JOIN ' . $this->_commentReplysTable . ' re ON cn.commentid = re.commentid  ' . 'LEFT JOIN ' . $this->_memebrsTable . ' m ON m.uid = cn.uid ' . 'WHERE cn.commentid IN (' . $this->_getImplodeString($ids) . ') ORDER BY postdate DESC';
		$result = $this->_db->query($sql);
		return $this->_getAllResultFromQuery($result, 'commentid');
	}
	
	/**
	 * 根据评论的ID列表，获取该条评论的子级评论详细信息表根据时间正序来
	 * @param array $rootIds 需要获取子级评论的根级评论ID
	 * @return array array(array(),array()...) 返回该根级目录下的所有子级的评论的信息
	 */
	function getChildCommentInfoList($rootIds) {
		if (!S::isArray($rootIds)) return array();
		$sql = "SELECT cn.*, re.parentid as parentid, re.rootid as rootid, m.icon as icon " . "FROM " . $this->_tableName . ' cn ' . 'LEFT JOIN ' . $this->_commentReplysTable . ' re ON cn.commentid = re.commentid ' . 'LEFT JOIN ' . $this->_memebrsTable . ' m ON m.uid = cn.uid ' . 'WHERE re.rootid IN (' . $this->_getImplodeString($rootIds) . ')' . ' ORDER BY cn.postdate ASC';
		return $this->_getAllResultFromQuery($this->_db->query($sql));
	}
	
	/**
	 * 根据对象ID和商品ID获得该属性下所有的店主回复(根据时间正序排序)
	 * 数组的key值用parentid来组合
	 * @param int $objectid 店铺ID
	 * @param int $typeid 商品ID
	 * @param int $start 开始位置
	 * @param int $perpage 查询条数
	 * @return array  array(array(),array()...) 返回该根级目录下的所有回复评论的评论的信息
	 */
	function getReplyCommentList($objectid, $typeid = 0) {
		if (!$objectid) return array();
		$_where = '';
		if (intval($typeid) != 0) $_where .= ' AND typeid = ' . $this->_addSlashes(intval($typeid));
		$_order = " ORDER BY postdate ASC";
		$_sql = "SELECT cn.*, re.parentid as parentid, re.rootid as rootid, m.icon as icon " . "FROM " . $this->_tableName . ' cn ' . 'LEFT JOIN ' . $this->_commentReplysTable . ' re ON cn.commentid = re.commentid ' . 'LEFT JOIN ' . $this->_memebrsTable . ' m ON m.uid = cn.uid ' . 'WHERE cn.isanswer = 1 AND re.parentid != 0 AND cn.objectid = ' . $this->_addSlashes($objectid) . $_where . $_order;
		$_result = $this->_db->query($_sql);
		return $this->_getAllResultFromQuery($_result);
	}
	
	/**
	 * 检查数据合法性
	 * @param array $fieldData 输入的数据
	 * @return array 返回符合的数据集
	 */
	function _clearFieldData($fieldData) {
		$clearData = $this->_checkAllowField($fieldData, $this->getStruct());
		return $clearData;
	}
	
	/**
	 * 获取的是 pw_dianpu_commentcontents的数据库表结构
	 * @return array 返回数据库表的数据结构
	 */
	function getStruct() {
		return array('commentid', 'title', 'content', 'objectid', 'typeid', 'uid', 'username', 'postdate', 'isanswer', 
			'state');
	}
}